
import React, { useEffect, useState } from 'react';
import { HashRouter as Router, Routes, Route, Link, useNavigate, useLocation, Navigate } from 'react-router-dom';
import { User, UserRole } from './types';
import { AuthService } from './services/AuthService';
import { SeedService } from './services/SeedService';

// Pages
import HomePage from './pages/HomePage';
import MediatorDirectory from './pages/public/MediatorDirectory';
import MediatorProfileView from './pages/public/MediatorProfileView';
import BookingStatusPage from './pages/public/BookingStatusPage';
import AuthPage from './pages/AuthPage';
import InfoHowToEngage from './pages/public/InfoHowToEngage';
import InfoWhenToUse from './pages/public/InfoWhenToUse';
import InfoPreparation from './pages/public/InfoPreparation';
import AboutUsPage from './pages/public/AboutUsPage';
import TermsOfUsePage from './pages/public/TermsOfUsePage';
import RequestCallBackPage from './pages/public/RequestCallBack';
import BookingSuccessPage from './pages/public/BookingSuccessPage';

// FDR Pages
import FDRInitiation from './pages/fdr/FDRInitiation';
import FDRDashboard from './pages/fdr/FDRDashboard';
import FDRIntakeQueue from './pages/mediator/FDRIntakeQueue';

import MediatorDashboard from './pages/mediator/MediatorDashboard';
import ProfileBuilder from './pages/mediator/ProfileBuilder';
import RequestManager from './pages/mediator/RequestManager';
import MediatorAvailabilityPage from './pages/mediator/MediatorAvailabilityPage';

import AdminDashboard from './pages/admin/AdminDashboard';
import AuditLogPage from './pages/admin/AuditLogPage';

// Staff Pages
import StaffIntakeQueue from './pages/staff/StaffIntakeQueue';
import StaffIntakeDetail from './pages/staff/StaffIntakeDetail';

// Components
const Layout: React.FC<{ children: React.ReactNode; user: User | null; onLogout: () => void }> = ({ children, user, onLogout }) => {
  return (
    <div className="min-h-screen flex flex-col">
      <header className="bg-white border-b sticky top-0 z-50">
        <div className="max-w-7xl mx-auto px-4 sm:px-6 lg:px-8 h-16 flex items-center justify-between">
          <div className="flex items-center gap-8">
            <Link to="/" className="text-2xl font-bold text-indigo-600 flex items-center gap-2">
              <span className="bg-indigo-600 text-white w-8 h-8 rounded flex items-center justify-center text-xl italic">m</span>
              medi8
            </Link>
            <nav className="hidden md:flex items-center gap-6 text-sm font-medium text-slate-600">
              <Link to="/mediators" className="hover:text-indigo-600">Find a Mediator</Link>
              <Link to="/fdr/start" className="bg-indigo-50 text-indigo-700 px-3 py-1 rounded-md hover:bg-indigo-100">Family Law (FDR)</Link>
              <Link to="/about" className="hover:text-indigo-600">About Us</Link>
            </nav>
          </div>
          <div className="flex items-center gap-4">
            {user ? (
              <div className="flex items-center gap-4">
                <div className="text-right hidden sm:block">
                  <div className="text-sm font-semibold">{user.displayName}</div>
                  <div className="text-xs text-slate-500 uppercase tracking-wider font-bold">{user.role}</div>
                </div>
                <div className="flex gap-2">
                  {user.role === UserRole.MEDIATOR && (
                    <Link to="/dashboard" className="text-xs font-bold text-indigo-600 hover:underline px-2 py-1 bg-indigo-50 rounded">Dashboard</Link>
                  )}
                  <Link to="/fdr/my-cases" className="text-xs font-bold text-indigo-600 hover:underline px-2 py-1 bg-indigo-50 rounded">My FDR Cases</Link>
                  <button 
                    onClick={onLogout}
                    className="bg-slate-100 hover:bg-slate-200 text-slate-700 px-3 py-1.5 rounded-md text-sm font-medium transition-colors"
                  >
                    Logout
                  </button>
                </div>
              </div>
            ) : (
              <Link to="/auth" className="bg-indigo-600 hover:bg-indigo-700 text-white px-4 py-2 rounded-md text-sm font-medium transition-colors shadow-sm">
                Login
              </Link>
            )}
          </div>
        </div>
      </header>
      <main className="flex-grow">
        {children}
      </main>
      <footer className="bg-slate-900 text-slate-400 py-12 px-4 text-xs">
        <div className="max-w-7xl mx-auto flex flex-col md:flex-row justify-between items-center gap-8">
          <div>
            <Link to="/" className="text-white text-xl font-bold flex items-center gap-2 mb-2">
              <span className="bg-indigo-600 text-white w-6 h-6 rounded flex items-center justify-center text-sm italic">m</span>
              medi8
            </Link>
            <p className="text-sm">Empowering conflict resolution through technology.</p>
          </div>
          <div className="text-xs text-center md:text-right">
            <p>(C) 2026 Mediation Institute Pty Ltd ABN 43 166 175 962 All rights reserved.</p>
            <p className="mt-2 flex justify-center md:justify-end gap-3">
              <span className="hover:text-white cursor-pointer transition-colors">Privacy Policy</span>
              <span>•</span>
              <Link to="/terms" className="hover:text-white transition-colors">Terms of Use</Link>
            </p>
          </div>
        </div>
      </footer>
    </div>
  );
};

const ProtectedRoute: React.FC<{ children: React.ReactNode; allowedRoles?: UserRole[] }> = ({ children, allowedRoles }) => {
  const user = AuthService.getCurrentUser();
  const location = useLocation();
  if (!user) return <Navigate to="/auth" state={{ from: location }} replace />;
  if (allowedRoles && !allowedRoles.includes(user.role)) return <Navigate to="/" replace />;
  return <>{children}</>;
};

const App: React.FC = () => {
  const [user, setUser] = useState<User | null>(AuthService.getCurrentUser());

  useEffect(() => {
    SeedService.seedDatabase();
  }, []);

  const handleLogout = () => {
    AuthService.logout();
    setUser(null);
  };

  return (
    <Router>
      <Layout user={user} onLogout={handleLogout}>
        <Routes>
          <Route path="/" element={<HomePage />} />
          <Route path="/mediators" element={<MediatorDirectory />} />
          <Route path="/mediators/:id" element={<MediatorProfileView />} />
          <Route path="/booking/:id" element={<BookingStatusPage />} />
          <Route path="/booking-success" element={<BookingSuccessPage />} />
          <Route path="/auth" element={<AuthPage onLogin={setUser} />} />
          <Route path="/info/how-to-engage" element={<InfoHowToEngage />} />
          <Route path="/info/when-to-use" element={<InfoWhenToUse />} />
          <Route path="/info/preparation" element={<InfoPreparation />} />
          <Route path="/about" element={<AboutUsPage />} />
          <Route path="/terms" element={<TermsOfUsePage />} />
          <Route path="/request-callback" element={<RequestCallBackPage />} />

          {/* FDR Specialized Routes */}
          <Route path="/fdr/start" element={<FDRInitiation />} />
          <Route path="/fdr/my-cases" element={<FDRDashboard />} />
          <Route path="/fdr/cases/:id" element={<FDRDashboard />} />

          {/* Mediator Routes */}
          <Route path="/dashboard" element={<ProtectedRoute allowedRoles={[UserRole.MEDIATOR]}><MediatorDashboard /></ProtectedRoute>} />
          <Route path="/dashboard/profile" element={<ProtectedRoute allowedRoles={[UserRole.MEDIATOR]}><ProfileBuilder /></ProtectedRoute>} />
          <Route path="/dashboard/requests" element={<ProtectedRoute allowedRoles={[UserRole.MEDIATOR]}><RequestManager /></ProtectedRoute>} />
          <Route path="/dashboard/availability" element={<ProtectedRoute allowedRoles={[UserRole.MEDIATOR]}><MediatorAvailabilityPage /></ProtectedRoute>} />
          <Route path="/dashboard/fdr-intake" element={<ProtectedRoute allowedRoles={[UserRole.MEDIATOR]}><FDRIntakeQueue /></ProtectedRoute>} />

          {/* Staff Routes */}
          <Route path="/staff/intake" element={<ProtectedRoute allowedRoles={[UserRole.STAFF, UserRole.ADMIN]}><StaffIntakeQueue /></ProtectedRoute>} />
          <Route path="/staff/intake/:id" element={<ProtectedRoute allowedRoles={[UserRole.STAFF, UserRole.ADMIN]}><StaffIntakeDetail /></ProtectedRoute>} />

          {/* Admin Routes */}
          <Route path="/admin" element={<ProtectedRoute allowedRoles={[UserRole.ADMIN]}><AdminDashboard /></ProtectedRoute>} />
          <Route path="/admin/audit" element={<ProtectedRoute allowedRoles={[UserRole.ADMIN]}><AuditLogPage /></ProtectedRoute>} />
        </Routes>
      </Layout>
    </Router>
  );
};

export default App;
