
import { PricingModel, HourlyRateBasis, PriceBand } from '../types';

export const PRICE_BAND_THRESHOLDS = {
  BUDGET_MAX: 150000, // $1,500
  STANDARD_MAX: 350000, // $3,500
};

export interface EstimateParams {
  pricing: PricingModel;
  expectedHours: number;
  partyCount: number;
  travelKm?: number;
  accommodationNights?: number;
}

export const estimateTotalCents = (params: EstimateParams): number => {
  const { pricing, expectedHours, partyCount, travelKm = 0, accommodationNights = 0 } = params;

  // Defensive defaults for numeric fields
  const preMediationFee = pricing?.preMediationFeeCents || 0;
  const hourlyRate = pricing?.hourlyRateCents || 0;
  const perKm = pricing?.perKmCents || 0;
  const travelTimeRate = pricing?.travelTimeHourlyRateCents || 0;
  const nightRate = pricing?.estimatedPerNightCents || 0;

  let total = preMediationFee;

  // Hourly component
  if (pricing?.hourlyRateBasis === HourlyRateBasis.PER_PARTY) {
    total += hourlyRate * expectedHours * partyCount;
  } else {
    total += hourlyRate * expectedHours;
  }

  // Travel
  if (pricing?.travelEnabled && travelKm > (pricing?.includedKm || 0)) {
    const billableKm = travelKm - (pricing?.includedKm || 0);
    total += billableKm * perKm;
    if (travelTimeRate) {
      // Simplistic: assume 1 hour per 80km of billable travel for estimation
      total += (billableKm / 80) * travelTimeRate;
    }
  }

  // Accommodation
  if (pricing?.accommodationEnabled && nightRate) {
    total += accommodationNights * nightRate;
  }

  return Math.round(total);
};

export const getPriceBand = (totalCents: number): PriceBand => {
  if (totalCents < PRICE_BAND_THRESHOLDS.BUDGET_MAX) return PriceBand.BUDGET;
  if (totalCents <= PRICE_BAND_THRESHOLDS.STANDARD_MAX) return PriceBand.STANDARD;
  return PriceBand.PREMIUM;
};

export const getPricingSummary = (pricing: PricingModel): string => {
  if (!pricing) return 'Pricing unavailable';
  const base = `${pricing.hourlyRateBasis === HourlyRateBasis.PER_PARTY ? 'Per Party' : 'Total'} rate`;
  const hourly = (pricing.hourlyRateCents || 0) / 100;
  const setup = (pricing.preMediationFeeCents || 0) / 100;
  return `${base}: $${hourly}/hr + $${setup} setup (Inc. GST)`;
};
