
import { MediatorProfilePublic, PriceBand, MediatorStatus } from '../types';
import { estimateTotalCents, getPriceBand } from './pricing';

export interface SuggestCriteria {
  categoryId: string;
  expectedHours: number;
  partyCount: number;
  onlineOnly?: boolean;
  priceBand?: PriceBand;
  maxBudgetCents?: number;
}

export interface SuggestedMediator {
  profile: MediatorProfilePublic;
  estimatedTotalCents: number;
  priceBand: PriceBand;
}

export const suggestMediators = (
  mediators: MediatorProfilePublic[],
  criteria: SuggestCriteria
): SuggestedMediator[] => {
  const { categoryId, expectedHours, partyCount, onlineOnly, priceBand, maxBudgetCents } = criteria;

  return (mediators || [])
    .filter((m) => {
      // Defensive check: ensure profile is complete enough for calculation and display
      if (!m) return false;
      if (!m.pricing) return false;
      if (!Array.isArray(m.categories)) return false;
      return m.status === MediatorStatus.PUBLISHED;
    })
    .filter((m) => {
      if (categoryId === 'all') return true;
      return m.categories.includes(categoryId);
    })
    .filter((m) => (onlineOnly ? m.onlineAvailable : true))
    .map((m) => {
      const estimate = estimateTotalCents({
        pricing: m.pricing,
        expectedHours,
        partyCount,
      });
      return {
        profile: m,
        estimatedTotalCents: estimate,
        priceBand: getPriceBand(estimate),
      };
    })
    .filter((s) => (priceBand ? s.priceBand === priceBand : true))
    .filter((s) => (maxBudgetCents ? s.estimatedTotalCents <= maxBudgetCents : true))
    .sort((a, b) => a.estimatedTotalCents - b.estimatedTotalCents);
};
