
import React, { useState } from 'react';
import { useNavigate } from 'react-router-dom';
import { AuthService } from '../services/AuthService';
import { SeedService } from '../services/SeedService';
import { User } from '../types';

const AuthPage: React.FC<{ onLogin: (user: User) => void }> = ({ onLogin }) => {
  const [email, setEmail] = useState('');
  const [error, setError] = useState('');
  const [loading, setLoading] = useState(false);
  const navigate = useNavigate();

  const handleLogin = async (e: React.FormEvent) => {
    e.preventDefault();
    setError('');
    setLoading(true);

    try {
      const user = await AuthService.login(email);
      if (user) {
        onLogin(user);
        navigate('/');
      } else {
        setError('Login failed. Please try a demo account.');
      }
    } catch (err) {
      setError('An error occurred. Please try again.');
    } finally {
      setLoading(false);
    }
  };

  const handleQuickLogin = async (demoEmail: string) => {
    setEmail(demoEmail);
    const user = await AuthService.login(demoEmail);
    if (user) {
      onLogin(user);
      navigate('/');
    }
  };

  const handleReset = async () => {
    if (confirm('This will wipe all local data and restore the original demo state. Continue?')) {
      await SeedService.resetDatabase();
    }
  };

  return (
    <div className="max-w-md mx-auto py-20 px-4">
      <div className="bg-white p-8 rounded-xl shadow-lg border border-slate-100">
        <h2 className="text-2xl font-bold text-center mb-8">Sign In to medi8</h2>
        
        <form onSubmit={handleLogin} className="space-y-4">
          <div>
            <label className="block text-sm font-medium text-slate-700 mb-1">Email Address</label>
            <input 
              type="email" 
              value={email}
              onChange={(e) => setEmail(e.target.value)}
              className="w-full px-4 py-2 border rounded-lg focus:ring-2 focus:ring-indigo-600 outline-none"
              placeholder="you@example.com"
              required
            />
          </div>
          {error && <p className="text-red-500 text-sm">{error}</p>}
          <button 
            type="submit" 
            disabled={loading}
            className="w-full bg-indigo-600 hover:bg-indigo-700 text-white font-bold py-2 rounded-lg transition-colors disabled:opacity-50"
          >
            {loading ? 'Logging in...' : 'Sign In'}
          </button>
        </form>

        <div className="mt-10 pt-6 border-t border-slate-100">
          <h3 className="text-xs font-bold text-slate-400 uppercase tracking-widest mb-4">Demo Logins</h3>
          <div className="grid grid-cols-1 gap-2">
            <button onClick={() => handleQuickLogin('admin@medi8.com.au')} className="text-left text-sm p-3 border rounded hover:bg-slate-50 border-slate-200">
              <span className="font-bold text-indigo-600">Admin:</span> admin@medi8.com.au
            </button>
            <button onClick={() => handleQuickLogin('mediator1@medi8r.com.au')} className="text-left text-sm p-3 border rounded hover:bg-slate-50 border-slate-200">
              <span className="font-bold text-indigo-600">Mediator:</span> mediator1@medi8r.com.au
            </button>
            <button onClick={() => handleQuickLogin('public@demo.com')} className="text-left text-sm p-3 border rounded hover:bg-slate-50 border-slate-200">
              <span className="font-bold text-indigo-600">Public:</span> public@demo.com
            </button>
          </div>
        </div>

        <div className="mt-8 text-center">
          <button 
            onClick={handleReset}
            className="text-slate-400 text-xs hover:text-red-500 transition-colors"
          >
            Reset Demo Data to Factory Settings
          </button>
        </div>
      </div>
    </div>
  );
};

export default AuthPage;
