
import React, { useState, useEffect } from 'react';
import { Link } from 'react-router-dom';
import { MediatorService } from '../../services/MediatorService';
import { AuthService } from '../../services/AuthService';
import { AuditService } from '../../services/AuditService';
import { MediatorProfilePrivate, MediatorStatus } from '../../types';

const AdminDashboard: React.FC = () => {
  const currentUser = AuthService.getCurrentUser();
  const [mediators, setMediators] = useState<MediatorProfilePrivate[]>([]);
  const [loading, setLoading] = useState(true);
  const [editingAlias, setEditingAlias] = useState<{id: string, alias: string} | null>(null);

  useEffect(() => {
    loadMediators();
  }, []);

  const loadMediators = async () => {
    const data = await MediatorService.getAllMediators();
    setMediators(data);
    setLoading(false);
  };

  const handleStatusChange = async (id: string, status: MediatorStatus) => {
    if (!currentUser) return;
    const med = mediators.find(m => m.id === id);
    await MediatorService.updateStatus(id, status);
    
    await AuditService.recordAction(
      currentUser, 
      status === MediatorStatus.PUBLISHED ? 'APPROVE_MEDIATOR' : 'SUSPEND_MEDIATOR',
      id,
      `Changed status of ${med?.displayName} to ${status}`
    );
    
    loadMediators();
  };

  const handleSaveAlias = async () => {
    if (editingAlias && currentUser) {
      const med = mediators.find(m => m.id === editingAlias.id);
      await MediatorService.assignAlias(editingAlias.id, editingAlias.alias);
      
      await AuditService.recordAction(
        currentUser,
        'ASSIGN_ALIAS',
        editingAlias.id,
        `Assigned alias ${editingAlias.alias} to ${med?.displayName}`
      );
      
      setEditingAlias(null);
      loadMediators();
    }
  };

  if (loading) return <div className="p-20 text-center text-slate-500">Loading administrative panel...</div>;

  const pending = mediators.filter(m => m.status === MediatorStatus.PENDING_APPROVAL);
  const published = mediators.filter(m => m.status === MediatorStatus.PUBLISHED);

  return (
    <div className="max-w-7xl mx-auto px-4 py-12">
      <div className="flex flex-col md:flex-row justify-between items-start md:items-end mb-10 gap-6">
        <div>
          <h1 className="text-3xl font-bold">Admin Control Center</h1>
          <p className="text-slate-500">Manage mediator profiles and system integrity</p>
        </div>
        <Link to="/admin/audit" className="bg-white border border-slate-200 text-slate-700 px-6 py-2 rounded-lg font-bold hover:bg-slate-50 shadow-sm transition-colors flex items-center gap-2">
          <svg xmlns="http://www.w3.org/2000/svg" className="h-5 w-5" viewBox="0 0 20 20" fill="currentColor">
            <path fillRule="evenodd" d="M4 4a2 2 0 012-2h4.586A2 2 0 0112 2.586L15.414 6A2 2 0 0116 7.414V16a2 2 0 01-2 2H6a2 2 0 01-2-2V4zm2 6a1 1 0 011-1h6a1 1 0 110 2H7a1 1 0 01-1-1zm1 3a1 1 0 100 2h6a1 1 0 100-2H7z" clipRule="evenodd" />
          </svg>
          Audit History
        </Link>
      </div>

      <div className="grid md:grid-cols-3 gap-6 mb-12">
        <div className="bg-indigo-600 text-white p-6 rounded-2xl shadow-xl">
          <div className="text-xs font-bold text-indigo-300 uppercase mb-2 tracking-widest">Pending Review</div>
          <div className="text-4xl font-extrabold">{pending.length}</div>
        </div>
        <div className="bg-white p-6 rounded-2xl border border-slate-200 shadow-sm">
          <div className="text-xs font-bold text-slate-400 uppercase mb-2 tracking-widest">Active Marketplace</div>
          <div className="text-4xl font-extrabold text-green-600">{published.length}</div>
        </div>
        <div className="bg-white p-6 rounded-2xl border border-slate-200 shadow-sm">
          <div className="text-xs font-bold text-slate-400 uppercase mb-2 tracking-widest">Total Profiles</div>
          <div className="text-4xl font-extrabold text-slate-900">{mediators.length}</div>
        </div>
      </div>

      <div className="bg-white rounded-2xl border border-slate-200 shadow-sm overflow-hidden">
        <div className="px-6 py-4 bg-slate-50 border-b border-slate-100 flex justify-between items-center">
          <h2 className="font-bold text-slate-700 uppercase text-xs tracking-wider">Mediator Registry</h2>
          <span className="text-[10px] text-slate-400 italic">Showing all registered accounts</span>
        </div>
        <div className="overflow-x-auto">
          <table className="w-full text-left">
            <thead className="bg-slate-50 text-slate-400 text-[10px] uppercase font-bold border-b border-slate-100">
              <tr>
                <th className="px-6 py-4">Mediator</th>
                <th className="px-6 py-4">Status</th>
                <th className="px-6 py-4">Marketplace Alias</th>
                <th className="px-6 py-4 text-right">Actions</th>
              </tr>
            </thead>
            <tbody className="divide-y divide-slate-50">
              {mediators.length === 0 ? (
                <tr><td colSpan={4} className="p-12 text-center text-slate-400 italic">No mediators registered.</td></tr>
              ) : (
                mediators.map(m => (
                  <tr key={m.id} className="hover:bg-slate-50 transition-colors">
                    <td className="px-6 py-4">
                      <div className="flex items-center gap-3">
                        <img src={m.photoUrl} className="w-10 h-10 rounded-full border-2 border-slate-100 object-cover" />
                        <div>
                          <div className="font-bold text-slate-900">{m.displayName}</div>
                          <div className="text-[10px] font-mono text-slate-400">{m.forwardToEmail}</div>
                        </div>
                      </div>
                    </td>
                    <td className="px-6 py-4">
                      <span className={`px-2 py-0.5 rounded text-[10px] font-bold tracking-wider ${
                        m.status === 'PUBLISHED' ? 'bg-green-100 text-green-700' : 
                        m.status === 'PENDING_APPROVAL' ? 'bg-amber-100 text-amber-700' : 'bg-red-100 text-red-700'
                      }`}>
                        {m.status}
                      </span>
                    </td>
                    <td className="px-6 py-4">
                      {editingAlias?.id === m.id ? (
                        <div className="flex gap-2">
                          <input 
                            value={editingAlias.alias} 
                            onChange={(e) => setEditingAlias({...editingAlias, alias: e.target.value})}
                            className="border text-xs p-1 rounded w-full outline-none focus:ring-1 focus:ring-indigo-500"
                            placeholder="jane.doe@medi8r.com.au"
                          />
                          <button onClick={handleSaveAlias} className="text-green-600 text-xs font-bold hover:underline">Save</button>
                        </div>
                      ) : (
                        <div className="flex items-center justify-between gap-2 max-w-[200px]">
                          <span className="text-xs font-mono text-slate-500 truncate">{m.publicEmail || 'not assigned'}</span>
                          <button onClick={() => setEditingAlias({id: m.id, alias: m.publicEmail || ''})} className="text-indigo-600 text-xs font-bold hover:underline">Edit</button>
                        </div>
                      )}
                    </td>
                    <td className="px-6 py-4 text-right">
                      <div className="flex justify-end gap-2">
                        {m.status !== MediatorStatus.PUBLISHED && (
                          <button 
                            onClick={() => handleStatusChange(m.id, MediatorStatus.PUBLISHED)}
                            className="bg-green-600 text-white text-[10px] px-3 py-1.5 rounded-lg font-bold uppercase hover:bg-green-700 transition-colors"
                          >
                            Approve
                          </button>
                        )}
                        {m.status !== MediatorStatus.SUSPENDED && (
                          <button 
                            onClick={() => handleStatusChange(m.id, MediatorStatus.SUSPENDED)}
                            className="bg-red-50 text-red-600 text-[10px] px-3 py-1.5 rounded-lg font-bold uppercase hover:bg-red-600 hover:text-white transition-all"
                          >
                            Suspend
                          </button>
                        )}
                      </div>
                    </td>
                  </tr>
                ))
              )}
            </tbody>
          </table>
        </div>
      </div>
    </div>
  );
};

export default AdminDashboard;
