
import React, { useState, useEffect } from 'react';
import { useParams, Link } from 'react-router-dom';
import { FDRService } from '../../services/FDRService';
import { MediatorService } from '../../services/MediatorService';
import { AuthService } from '../../services/AuthService';
import { FDRCase, FDRStage, ScreeningOutcome, MediatorProfilePublic } from '../../types';

const FDRDashboard: React.FC = () => {
  const { id } = useParams<{ id: string }>();
  const user = AuthService.getCurrentUser();
  const [fdrCase, setFdrCase] = useState<FDRCase | null>(null);
  const [availableMediators, setAvailableMediators] = useState<MediatorProfilePublic[]>([]);
  const [loading, setLoading] = useState(true);

  useEffect(() => {
    if (id) loadData(id);
    else if (user) loadAllCases(user.id);
  }, [id, user]);

  const loadAllCases = async (uid: string) => {
    const cases = await FDRService.getCasesForUser(uid);
    if (cases.length > 0 && !id) {
       // Just set the first one for simplicity in this demo view if no ID provided
       setFdrCase(cases[0]);
    }
    setLoading(false);
  };

  const loadData = async (caseId: string) => {
    const c = await FDRService.getCase(caseId);
    if (c) {
      setFdrCase(c);
      const meds = await MediatorService.getPublishedMediators();
      setAvailableMediators(meds.filter(m => m.categories.includes('cat-family')));
    }
    setLoading(false);
  };

  const handlePropose = async (medIds: string[]) => {
    if (!fdrCase) return;
    const selection = { ...fdrCase.selection, p1ProposedIds: medIds };
    await FDRService.updateSelection(fdrCase.id, selection);
    loadData(fdrCase.id);
  };

  const handleP2Select = async (medId: string) => {
    if (!fdrCase) return;
    const selection = { ...fdrCase.selection, selectedMediatorId: medId };
    await FDRService.updateSelection(fdrCase.id, selection);
    loadData(fdrCase.id);
  };

  if (loading) return <div className="p-20 text-center">Loading case details...</div>;
  if (!fdrCase) return <div className="p-20 text-center text-slate-500">No active FDR cases found. <Link to="/fdr/start" className="text-indigo-600 underline">Start one now.</Link></div>;

  const isParty1 = user?.id === fdrCase.party1Id;
  const isParty2 = user?.email === fdrCase.party2Email || user?.id === fdrCase.party2Id;

  return (
    <div className="max-w-6xl mx-auto py-12 px-4">
      <div className="flex justify-between items-end mb-8">
        <div>
          <h1 className="text-3xl font-bold">FDR Case Manager</h1>
          <p className="text-slate-500">Ref: {fdrCase.id} • {fdrCase.disputeType} Dispute</p>
        </div>
        <div className="bg-white px-4 py-2 rounded-lg border shadow-sm">
          <span className="text-[10px] uppercase font-bold text-slate-400 block mb-0.5">Current Stage</span>
          <span className="text-indigo-600 font-bold">{fdrCase.stage}</span>
        </div>
      </div>

      <div className="grid lg:grid-cols-4 gap-4 mb-12">
        {Object.values(FDRStage).map((s, i) => (
          <div key={s} className={`p-4 rounded-xl border-2 text-center transition-all ${
            fdrCase.stage === s ? 'border-indigo-600 bg-indigo-50 text-indigo-700 font-bold' : 
            i < Object.values(FDRStage).indexOf(fdrCase.stage) ? 'border-green-100 bg-green-50 text-green-700' : 'border-slate-100 text-slate-300'
          }`}>
            <div className="text-[10px] uppercase tracking-widest mb-1">Stage {i+1}</div>
            <div className="text-sm">{s.replace('_', ' ')}</div>
          </div>
        ))}
      </div>

      <div className="grid lg:grid-cols-3 gap-8">
        <div className="lg:col-span-2 space-y-8">
          {/* Stage 1 Content */}
          {fdrCase.stage === FDRStage.SCREENING && (
            <div className="bg-white p-8 rounded-2xl border border-slate-200 shadow-sm text-center">
              <div className="w-16 h-16 bg-indigo-100 text-indigo-600 rounded-full flex items-center justify-center mx-auto mb-6">
                <svg className="h-8 w-8 animate-pulse" fill="none" viewBox="0 0 24 24" stroke="currentColor">
                  <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M9 12l2 2 4-4m5.618-4.016A11.955 11.955 0 0112 2.944a11.955 11.955 0 01-8.618 3.04A12.02 12.02 0 003 9c0 5.591 3.824 10.29 9 11.622 5.176-1.332 9-6.03 9-11.622 0-1.042-.133-2.052-.382-3.016z" />
                </svg>
              </div>
              <h2 className="text-2xl font-bold mb-4">Screening in Progress</h2>
              <p className="text-slate-600 mb-8 max-w-md mx-auto">
                A Junior FDRP officer is reviewing your declarations. We will update the outcome shortly. 
                Until this gate is cleared, Party 2 will not be contacted.
              </p>
              <div className="bg-slate-50 p-4 rounded-xl text-left">
                <h4 className="text-xs font-bold text-slate-400 uppercase mb-3 tracking-widest">Submitted Data</h4>
                <div className="space-y-2 text-sm">
                  <div className="flex justify-between"><span>Family Violence:</span> <span className="font-bold">{fdrCase.declarations.familyViolence ? 'Yes' : 'No'}</span></div>
                  <div className="flex justify-between"><span>Intervention Orders:</span> <span className="font-bold">{fdrCase.declarations.interventionOrders ? 'Yes' : 'No'}</span></div>
                  <div className="flex justify-between"><span>Pending Court:</span> <span className="font-bold">{fdrCase.declarations.pendingProceedings ? 'Yes' : 'No'}</span></div>
                </div>
              </div>
            </div>
          )}

          {/* Stage 2 Content: Selection */}
          {fdrCase.stage === FDRStage.SELECTION && (
            <div className="bg-white p-8 rounded-2xl border border-slate-200 shadow-sm">
              <h2 className="text-2xl font-bold mb-6">Mediator Selection</h2>
              
              {isParty1 && fdrCase.selection.p1ProposedIds.length === 0 && (
                <div className="space-y-6">
                  <p className="text-slate-600">Screening cleared. Please propose up to 3 eligible mediators for the other party to review.</p>
                  <div className="grid md:grid-cols-2 gap-4">
                    {availableMediators.slice(0, 4).map(m => (
                      <div key={m.id} className="p-4 border rounded-xl flex items-center gap-4">
                        <img src={m.photoUrl} className="w-12 h-12 rounded-full object-cover" />
                        <div className="flex-grow">
                          <div className="font-bold text-sm">{m.displayName}</div>
                          <div className="text-[10px] text-slate-400">{m.headline}</div>
                        </div>
                        <button 
                          onClick={() => handlePropose([m.id])}
                          className="bg-indigo-600 text-white text-[10px] px-3 py-1.5 rounded font-bold"
                        >
                          Propose
                        </button>
                      </div>
                    ))}
                  </div>
                </div>
              )}

              {isParty2 && fdrCase.selection.p1ProposedIds.length > 0 && !fdrCase.selection.selectedMediatorId && (
                <div className="space-y-6">
                   <div className="p-4 bg-indigo-50 border border-indigo-100 rounded-xl">
                      <h3 className="font-bold text-indigo-900 mb-2">Party 1 Proposals</h3>
                      <div className="space-y-3">
                        {fdrCase.selection.p1ProposedIds.map(pid => {
                          const med = availableMediators.find(m => m.id === pid);
                          return (
                            <div key={pid} className="flex justify-between items-center bg-white p-3 rounded shadow-sm border border-indigo-100">
                               <div className="font-bold text-sm">{med?.displayName}</div>
                               <button onClick={() => handleP2Select(pid)} className="bg-indigo-600 text-white text-[10px] px-4 py-1 rounded-full font-bold">Select</button>
                            </div>
                          );
                        })}
                      </div>
                   </div>
                   <div className="text-center">
                      <span className="text-slate-400 text-xs">OR</span>
                      <button className="block w-full mt-4 text-indigo-600 font-bold hover:underline">Reject and Propose Alternatives</button>
                   </div>
                </div>
              )}

              {!isParty1 && !isParty2 && (
                <div className="p-8 text-center text-slate-400 italic">Waiting for party interaction...</div>
              )}
            </div>
          )}

          {/* Stage 3 & 4 (Simplified for brevity) */}
          {(fdrCase.stage === FDRStage.PRE_MEDIATION || fdrCase.stage === FDRStage.MEDIATION) && (
            <div className="bg-white p-8 rounded-2xl border border-slate-200 shadow-sm">
               <h2 className="text-2xl font-bold mb-4">Mediation Preparation</h2>
               <div className="p-10 text-center space-y-4">
                  <div className="w-16 h-16 bg-green-100 text-green-600 rounded-full flex items-center justify-center mx-auto">
                    <svg className="h-8 w-8" fill="none" viewBox="0 0 24 24" stroke="currentColor"><path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M5 13l4 4L19 7" /></svg>
                  </div>
                  <p className="text-slate-600">The selected mediator will now contact each party separately for substantive pre-mediation assessment.</p>
                  <div className="text-sm font-bold text-indigo-600">Selected: {availableMediators.find(m => m.id === fdrCase.selection.selectedMediatorId)?.displayName}</div>
               </div>
            </div>
          )}
        </div>

        <div className="space-y-4">
           <div className="bg-slate-900 text-white p-6 rounded-2xl shadow-xl">
              <h3 className="font-bold text-xs uppercase tracking-widest text-slate-500 mb-4">Case Status</h3>
              <div className="space-y-3 text-sm">
                <div className="flex justify-between"><span className="text-slate-500">Party 1:</span> <span className="font-mono text-xs">{fdrCase.party1Email}</span></div>
                <div className="flex justify-between"><span className="text-slate-500">Party 2:</span> <span className="font-mono text-xs">{fdrCase.party2Email || 'Not Invited'}</span></div>
                <div className="pt-4 border-t border-white/10">
                   <div className="text-green-400 font-bold mb-1 flex items-center gap-1">
                      <svg className="h-4 w-4" fill="currentColor" viewBox="0 0 20 20"><path fillRule="evenodd" d="M10 18a8 8 0 100-16 8 8 0 000 16zm3.707-9.293a1 1 0 00-1.414-1.414L9 10.586 7.707 9.293a1 1 0 00-1.414 1.414l2 2a1 1 0 001.414 0l4-4z" clipRule="evenodd" /></svg>
                      Screening Fee Paid
                   </div>
                </div>
              </div>
           </div>

           <div className="bg-white p-6 rounded-2xl border border-slate-100 shadow-sm">
              <h3 className="font-bold text-xs uppercase text-slate-400 mb-4">Resources</h3>
              <ul className="space-y-3 text-xs">
                <li><Link to="/info/how-to-engage" className="text-indigo-600 hover:underline">FDR Process Guide</Link></li>
                <li><a href="#" className="text-indigo-600 hover:underline">Understanding s60I Certificates</a></li>
                <li><a href="#" className="text-indigo-600 hover:underline">Safety Planning Support</a></li>
              </ul>
           </div>
        </div>
      </div>
    </div>
  );
};

export default FDRDashboard;
