
import React, { useState } from 'react';
import { useNavigate } from 'react-router-dom';
import { AuthService } from '../../services/AuthService';
import { FDRService } from '../../services/FDRService';
import { FDRCase, FDRStage } from '../../types';

const FDRInitiation: React.FC = () => {
  const user = AuthService.getCurrentUser();
  const navigate = useNavigate();
  const [step, setStep] = useState(0);
  const [form, setForm] = useState({
    disputeType: 'PARENTING' as FDRCase['disputeType'],
    p2Email: '',
    fv: false,
    intervention: false,
    pending: false,
    docName: ''
  });
  const [loading, setLoading] = useState(false);

  if (!user) {
    return (
      <div className="max-w-xl mx-auto py-20 px-4 text-center">
        <h2 className="text-2xl font-bold mb-4">Sign in required</h2>
        <p className="mb-6 text-slate-500">You must be signed in to initiate a Family Dispute Resolution matter.</p>
        <button onClick={() => navigate('/auth')} className="bg-indigo-600 text-white px-8 py-3 rounded-lg font-bold">Sign In Now</button>
      </div>
    );
  }

  const handlePayAndSubmit = async () => {
    setLoading(true);
    const caseId = 'fdr-' + Math.random().toString(36).substr(2, 9);
    const newCase: FDRCase = {
      id: caseId,
      party1Id: user.id,
      party1Email: user.email,
      party2Email: form.p2Email,
      disputeType: form.disputeType,
      stage: FDRStage.SCREENING,
      timestamp: Date.now(),
      screeningFeePaid: true,
      declarations: {
        familyViolence: form.fv,
        interventionOrders: form.intervention,
        pendingProceedings: form.pending
      },
      documents: form.docName ? [{ type: 'other', url: '#', name: form.docName }] : [],
      selection: {
        round: 1,
        p1ProposedIds: [],
        p2ProposedIds: [],
        selectionFailed: false
      },
      preMediationCompleted: false
    };

    await FDRService.createCase(newCase);
    navigate(`/fdr/cases/${caseId}`);
  };

  return (
    <div className="max-w-4xl mx-auto py-12 px-4">
      <div className="bg-white rounded-2xl shadow-xl overflow-hidden border border-slate-100">
        <div className="bg-indigo-600 p-8 text-white">
          <h1 className="text-3xl font-bold mb-2 uppercase tracking-tight">Family Law FDR Initiation</h1>
          <p className="text-indigo-100 opacity-80">Stage 1: Administrative Screening Intake</p>
        </div>

        <div className="p-8">
          {step === 0 && (
            <div className="space-y-6 animate-in fade-in duration-300">
              <div className="bg-amber-50 p-6 rounded-xl border border-amber-100 text-amber-800 text-sm leading-relaxed">
                <h3 className="font-bold mb-2">Important Disclosures</h3>
                <ul className="list-disc pl-4 space-y-1">
                  <li>Screening is preliminary and administrative only.</li>
                  <li>It may result in no contact with the other party if safety risks are identified.</li>
                  <li>Payment of the screening fee does not guarantee mediation will proceed.</li>
                </ul>
              </div>

              <div>
                <label className="block text-sm font-bold text-slate-700 mb-2">Nature of Dispute</label>
                <div className="grid grid-cols-3 gap-4">
                  {(['PARENTING', 'PROPERTY', 'BOTH'] as const).map(type => (
                    <button
                      key={type}
                      onClick={() => setForm({...form, disputeType: type})}
                      className={`py-3 rounded-lg border-2 font-bold text-sm transition-all ${
                        form.disputeType === type ? 'border-indigo-600 bg-indigo-50 text-indigo-700' : 'border-slate-100 text-slate-400 hover:border-slate-200'
                      }`}
                    >
                      {type}
                    </button>
                  ))}
                </div>
              </div>

              <div>
                <label className="block text-sm font-bold text-slate-700 mb-1">Other Party's Email (Optional at this stage)</label>
                <input 
                  type="email"
                  value={form.p2Email}
                  onChange={e => setForm({...form, p2Email: e.target.value})}
                  className="w-full border rounded-lg p-3 outline-none focus:ring-2 focus:ring-indigo-600"
                  placeholder="other.party@example.com"
                />
                <p className="text-[10px] text-slate-400 mt-1 uppercase font-bold">No contact will be made until screening is cleared.</p>
              </div>

              <button onClick={() => setStep(1)} className="w-full bg-indigo-600 text-white font-bold py-4 rounded-xl shadow-lg hover:bg-indigo-700 transition-colors">
                Continue to Declarations
              </button>
            </div>
          )}

          {step === 1 && (
            <div className="space-y-6 animate-in slide-in-from-right-4 duration-300">
              <h2 className="text-xl font-bold border-b pb-4">Safety & Legal Declarations</h2>
              <div className="space-y-4">
                <label className="flex items-start gap-3 p-4 bg-slate-50 rounded-xl cursor-pointer hover:bg-slate-100 transition-colors">
                  <input 
                    type="checkbox" checked={form.fv} 
                    onChange={e => setForm({...form, fv: e.target.checked})}
                    className="w-5 h-5 mt-0.5 text-indigo-600"
                  />
                  <div>
                    <span className="font-bold text-slate-700 block text-sm">Family Violence</span>
                    <span className="text-xs text-slate-500">I declare there are history or current risks of family violence.</span>
                  </div>
                </label>

                <label className="flex items-start gap-3 p-4 bg-slate-50 rounded-xl cursor-pointer hover:bg-slate-100 transition-colors">
                  <input 
                    type="checkbox" checked={form.intervention} 
                    onChange={e => setForm({...form, intervention: e.target.checked})}
                    className="w-5 h-5 mt-0.5 text-indigo-600"
                  />
                  <div>
                    <span className="font-bold text-slate-700 block text-sm">Intervention / Protection Orders</span>
                    <span className="text-xs text-slate-500">I declare there are active intervention orders between parties.</span>
                  </div>
                </label>

                <label className="flex items-start gap-3 p-4 bg-slate-50 rounded-xl cursor-pointer hover:bg-slate-100 transition-colors">
                  <input 
                    type="checkbox" checked={form.pending} 
                    onChange={e => setForm({...form, pending: e.target.checked})}
                    className="w-5 h-5 mt-0.5 text-indigo-600"
                  />
                  <div>
                    <span className="font-bold text-slate-700 block text-sm">Pending Court Proceedings</span>
                    <span className="text-xs text-slate-500">I declare there are active family law proceedings in court.</span>
                  </div>
                </label>
              </div>

              <div>
                <label className="block text-sm font-bold text-slate-700 mb-2">Upload Relevant Orders (Simulation)</label>
                <input 
                  type="text"
                  placeholder="Enter filename of order (e.g. Orders_2023.pdf)"
                  value={form.docName}
                  onChange={e => setForm({...form, docName: e.target.value})}
                  className="w-full border rounded-lg p-3 text-sm"
                />
              </div>

              <div className="flex justify-between pt-6">
                <button onClick={() => setStep(0)} className="text-slate-500 font-bold px-4">Back</button>
                <button onClick={() => setStep(2)} className="bg-indigo-600 text-white px-8 py-3 rounded-lg font-bold shadow-lg">
                  Next: Payment
                </button>
              </div>
            </div>
          )}

          {step === 2 && (
            <div className="text-center space-y-6 animate-in zoom-in-95 duration-300 py-10">
              <div className="w-20 h-20 bg-indigo-100 text-indigo-600 rounded-full flex items-center justify-center mx-auto mb-4">
                <svg className="h-10 w-10" fill="none" viewBox="0 0 24 24" stroke="currentColor">
                  <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M12 8c-1.657 0-3 .895-3 2s1.343 2 3 2 3 .895 3 2-1.343 2-3 2m0-8c1.11 0 2.08.402 2.599 1M12 8V7m0 1v8m0 0v1m0-1c-1.11 0-2.08-.402-2.599-1M21 12a9 9 0 11-18 0 9 9 0 0118 0z" />
                </svg>
              </div>
              <h2 className="text-3xl font-bold">Administrative Fee</h2>
              <p className="text-slate-500 max-w-sm mx-auto">A fee of <strong>$110.00</strong> is required for initial screening by a Junior FDRP officer.</p>
              
              <div className="bg-slate-50 p-6 rounded-2xl border border-slate-200 inline-block text-left w-full max-w-sm">
                <div className="flex justify-between font-bold text-slate-700">
                  <span>Intake Screening</span>
                  <span>$110.00</span>
                </div>
                <div className="text-[10px] text-slate-400 mt-2 uppercase tracking-wider font-bold">Includes 15-min online assessment</div>
              </div>

              <div className="pt-6">
                <button 
                  disabled={loading}
                  onClick={handlePayAndSubmit}
                  className="w-full max-w-sm bg-green-600 hover:bg-green-700 text-white font-bold py-4 rounded-xl shadow-xl transition-all active:scale-95"
                >
                  {loading ? 'Processing...' : 'Pay & Initiate Screening'}
                </button>
              </div>
            </div>
          )}
        </div>
      </div>
    </div>
  );
};

export default FDRInitiation;
