
import React, { useState, useEffect } from 'react';
import { FDRService } from '../../services/FDRService';
import { FDRCase, ScreeningOutcome } from '../../types';

const FDRIntakeQueue: React.FC = () => {
  const [cases, setCases] = useState<FDRCase[]>([]);
  const [loading, setLoading] = useState(true);
  const [selectedCase, setSelectedCase] = useState<FDRCase | null>(null);

  useEffect(() => {
    loadQueue();
  }, []);

  const loadQueue = async () => {
    setLoading(true);
    const data = await FDRService.getPendingScreenings();
    setCases(data);
    setLoading(false);
  };

  const handleUpdate = async (outcome: ScreeningOutcome) => {
    if (!selectedCase) return;
    await FDRService.updateOutcome(selectedCase.id, outcome);
    setSelectedCase(null);
    loadQueue();
  };

  if (loading) return <div className="p-20 text-center">Loading intake queue...</div>;

  return (
    <div className="max-w-7xl mx-auto py-12 px-4">
      <h1 className="text-3xl font-bold mb-8">FDR Intake Queue (Administrative)</h1>
      
      <div className="grid lg:grid-cols-3 gap-8">
        <div className="lg:col-span-1 space-y-4">
          <div className="bg-white rounded-xl border border-slate-200 overflow-hidden">
            <div className="px-6 py-4 bg-slate-50 border-b font-bold text-xs uppercase tracking-widest text-slate-500">
              Pending Screenings ({cases.length})
            </div>
            <div className="divide-y divide-slate-100">
              {cases.length === 0 ? (
                <div className="p-8 text-center text-slate-400 italic">No pending screenings</div>
              ) : (
                cases.map(c => (
                  <button 
                    key={c.id} 
                    onClick={() => setSelectedCase(c)}
                    className={`w-full text-left p-4 hover:bg-slate-50 transition-colors ${selectedCase?.id === c.id ? 'bg-indigo-50 ring-1 ring-inset ring-indigo-200' : ''}`}
                  >
                    <div className="font-bold text-sm">{c.party1Email}</div>
                    <div className="text-[10px] text-slate-400 uppercase tracking-tighter mt-1">{c.disputeType} • {new Date(c.timestamp).toLocaleDateString('en-AU', { day: 'numeric', month: 'short', year: 'numeric' })}</div>
                  </button>
                ))
              )}
            </div>
          </div>
        </div>

        <div className="lg:col-span-2">
          {selectedCase ? (
            <div className="bg-white rounded-2xl border border-slate-200 shadow-xl overflow-hidden animate-in fade-in slide-in-from-bottom-4">
              <div className="p-8 space-y-8">
                <div>
                  <h2 className="text-2xl font-bold mb-2">Screening Assessment</h2>
                  <p className="text-slate-500 text-sm">Reviewing application for <strong>{selectedCase.party1Email}</strong></p>
                </div>

                <div className="grid md:grid-cols-2 gap-8">
                  <div className="space-y-4">
                    <h3 className="text-xs font-bold uppercase tracking-widest text-slate-400">Declarations</h3>
                    <div className="space-y-3">
                      <div className={`p-3 rounded-lg border flex justify-between items-center ${selectedCase.declarations.familyViolence ? 'bg-red-50 border-red-200 text-red-700' : 'bg-green-50 border-green-200 text-green-700'}`}>
                        <span className="text-sm">Family Violence Risk</span>
                        <span className="font-bold">{selectedCase.declarations.familyViolence ? 'YES' : 'NO'}</span>
                      </div>
                      <div className={`p-3 rounded-lg border flex justify-between items-center ${selectedCase.declarations.interventionOrders ? 'bg-red-50 border-red-200 text-red-700' : 'bg-green-50 border-green-200 text-green-700'}`}>
                        <span className="text-sm">Active IVOs</span>
                        <span className="font-bold">{selectedCase.declarations.interventionOrders ? 'YES' : 'NO'}</span>
                      </div>
                      <div className={`p-3 rounded-lg border flex justify-between items-center ${selectedCase.declarations.pendingProceedings ? 'bg-amber-50 border-amber-200 text-amber-700' : 'bg-green-50 border-green-200 text-green-700'}`}>
                        <span className="text-sm">Pending Court</span>
                        <span className="font-bold">{selectedCase.declarations.pendingProceedings ? 'YES' : 'NO'}</span>
                      </div>
                    </div>
                  </div>

                  <div className="space-y-4">
                    <h3 className="text-xs font-bold uppercase tracking-widest text-slate-400">Case Documents</h3>
                    <div className="bg-slate-50 p-4 rounded-xl border border-slate-200">
                      {selectedCase.documents.length === 0 ? (
                        <p className="text-xs text-slate-400 italic">No documents uploaded</p>
                      ) : (
                        <ul className="space-y-2">
                          {selectedCase.documents.map((d, i) => (
                            <li key={i} className="flex items-center gap-2 text-xs text-indigo-600 font-bold hover:underline cursor-pointer">
                              <svg className="h-4 w-4" fill="none" viewBox="0 0 24 24" stroke="currentColor"><path d="M7 21h10a2 2 0 002-2V9.414a1 1 0 00-.293-.707l-5.414-5.414A1 1 0 0012.586 3H7a2 2 0 00-2 2v14a2 2 0 002 2z" /></svg>
                              {d.name}
                            </li>
                          ))}
                        </ul>
                      )}
                    </div>
                  </div>
                </div>

                <div className="pt-8 border-t border-slate-100 flex flex-col md:flex-row gap-4">
                  <button 
                    onClick={() => handleUpdate(ScreeningOutcome.PROCEED)}
                    className="flex-grow bg-indigo-600 text-white font-bold py-4 rounded-xl shadow-lg hover:bg-indigo-700 transition-colors"
                  >
                    Outcome 1: Proceed to Selection
                  </button>
                  <button 
                    onClick={() => handleUpdate(ScreeningOutcome.ESCALATE)}
                    className="flex-grow bg-amber-500 text-white font-bold py-4 rounded-xl shadow-lg hover:bg-amber-600 transition-colors"
                  >
                    Outcome 2: Escalate to Senior
                  </button>
                  <button 
                    onClick={() => handleUpdate(ScreeningOutcome.NOT_APPROPRIATE)}
                    className="flex-grow bg-red-600 text-white font-bold py-4 rounded-xl shadow-lg hover:bg-red-700 transition-colors"
                  >
                    Outcome 3: Not Appropriate
                  </button>
                </div>
              </div>
            </div>
          ) : (
            <div className="h-full flex items-center justify-center p-20 text-center bg-slate-50 rounded-2xl border-2 border-dashed border-slate-200 text-slate-400">
              Select a case from the queue to perform Administrative Screening
            </div>
          )}
        </div>
      </div>
    </div>
  );
};

export default FDRIntakeQueue;
