import React, { useEffect, useState } from 'react';
import { AvailabilityService } from '../../services/AvailabilityService';
import { AvailabilityPeriod, MediatorAvailability } from '../../types';

interface Props {
  mediatorId: string;
}

const MediatorAvailabilityManager: React.FC<Props> = ({ mediatorId }) => {
  const [availability, setAvailability] = useState<MediatorAvailability | null>(null);
  const [loading, setLoading] = useState(true);
  const [showAddForm, setShowAddForm] = useState(false);
  
  // Form state
  const [startDate, setStartDate] = useState('');
  const [endDate, setEndDate] = useState('');
  const [reason, setReason] = useState('Holiday');
  const [notes, setNotes] = useState('');
  const [saving, setSaving] = useState(false);

  useEffect(() => {
    loadAvailability();
  }, [mediatorId]);

  const loadAvailability = async () => {
    setLoading(true);
    const avail = await AvailabilityService.getMediatorAvailability(mediatorId);
    setAvailability(avail);
    setLoading(false);
  };

  const handleAddPeriod = async () => {
    if (!startDate || !endDate) {
      alert('Please select both start and end dates');
      return;
    }

    const start = new Date(startDate).getTime();
    const end = new Date(endDate).getTime();

    if (end < start) {
      alert('End date must be after start date');
      return;
    }

    setSaving(true);
    await AvailabilityService.createUnavailablePeriod(
      mediatorId,
      start,
      end,
      reason,
      notes
    );
    
    // Reset form
    setStartDate('');
    setEndDate('');
    setReason('Holiday');
    setNotes('');
    setShowAddForm(false);
    setSaving(false);
    
    // Reload
    await loadAvailability();
  };

  const handleDeletePeriod = async (periodId: string) => {
    if (!confirm('Remove this unavailability period?')) return;
    
    await AvailabilityService.deletePeriod(periodId);
    await loadAvailability();
  };

  if (loading) {
    return (
      <div className="text-center py-8">
        <div className="inline-block w-8 h-8 border-4 border-indigo-600 border-t-transparent rounded-full animate-spin"></div>
        <p className="mt-4 text-slate-600">Loading availability...</p>
      </div>
    );
  }

  if (!availability) return null;

  return (
    <div className="space-y-6">
      {/* Current Status */}
      <div className="bg-white rounded-lg shadow p-6">
        <div className="flex items-center justify-between mb-4">
          <h2 className="text-2xl font-bold text-slate-900">Availability Status</h2>
          {availability.status === 'AVAILABLE' ? (
            <span className="px-4 py-2 bg-green-100 text-green-800 rounded-full font-semibold">
              ✓ Currently Available
            </span>
          ) : (
            <span className="px-4 py-2 bg-red-100 text-red-800 rounded-full font-semibold">
              ⚠ Currently Unavailable
            </span>
          )}
        </div>

        {availability.currentPeriod && (
          <div className="bg-yellow-50 border-l-4 border-yellow-500 p-4">
            <div className="flex items-start">
              <div className="flex-1">
                <p className="font-semibold text-slate-900">
                  You are marked as unavailable
                </p>
                <p className="text-slate-700 mt-1">
                  {AvailabilityService.formatDateRange(
                    availability.currentPeriod.startDate,
                    availability.currentPeriod.endDate
                  )}
                </p>
                {availability.currentPeriod.reason && (
                  <p className="text-sm text-slate-600 mt-1">
                    Reason: {availability.currentPeriod.reason}
                  </p>
                )}
              </div>
              <button
                onClick={() => handleDeletePeriod(availability.currentPeriod!.id)}
                className="ml-4 text-red-600 hover:text-red-700 text-sm font-medium"
              >
                End Now
              </button>
            </div>
          </div>
        )}

        <p className="text-slate-600 mt-4">
          When you're unavailable, you won't appear in client searches and cannot receive new bookings.
        </p>
      </div>

      {/* Add New Period */}
      <div className="bg-white rounded-lg shadow p-6">
        <div className="flex items-center justify-between mb-4">
          <h3 className="text-lg font-semibold text-slate-900">Manage Unavailability</h3>
          {!showAddForm && (
            <button
              onClick={() => setShowAddForm(true)}
              className="bg-indigo-600 hover:bg-indigo-700 text-white px-4 py-2 rounded-lg font-medium"
            >
              + Add Unavailable Period
            </button>
          )}
        </div>

        {showAddForm && (
          <div className="border border-slate-200 rounded-lg p-4 mb-4">
            <div className="grid grid-cols-1 md:grid-cols-2 gap-4 mb-4">
              <div>
                <label className="block text-sm font-medium text-slate-700 mb-2">
                  Start Date *
                </label>
                <input
                  type="date"
                  value={startDate}
                  onChange={(e) => setStartDate(e.target.value)}
                  min={new Date().toISOString().split('T')[0]}
                  className="w-full border border-slate-300 rounded-lg px-3 py-2"
                />
              </div>
              <div>
                <label className="block text-sm font-medium text-slate-700 mb-2">
                  End Date *
                </label>
                <input
                  type="date"
                  value={endDate}
                  onChange={(e) => setEndDate(e.target.value)}
                  min={startDate || new Date().toISOString().split('T')[0]}
                  className="w-full border border-slate-300 rounded-lg px-3 py-2"
                />
              </div>
            </div>

            <div className="mb-4">
              <label className="block text-sm font-medium text-slate-700 mb-2">
                Reason
              </label>
              <select
                value={reason}
                onChange={(e) => setReason(e.target.value)}
                className="w-full border border-slate-300 rounded-lg px-3 py-2"
              >
                <option value="Holiday">Holiday</option>
                <option value="Illness">Illness</option>
                <option value="Conference">Conference/Training</option>
                <option value="Personal">Personal Leave</option>
                <option value="Other">Other</option>
              </select>
            </div>

            <div className="mb-4">
              <label className="block text-sm font-medium text-slate-700 mb-2">
                Notes (Optional, Private)
              </label>
              <textarea
                value={notes}
                onChange={(e) => setNotes(e.target.value)}
                placeholder="Add any internal notes..."
                className="w-full border border-slate-300 rounded-lg px-3 py-2 min-h-[80px]"
              />
            </div>

            <div className="flex gap-3">
              <button
                onClick={handleAddPeriod}
                disabled={saving || !startDate || !endDate}
                className="bg-indigo-600 hover:bg-indigo-700 text-white px-4 py-2 rounded-lg font-medium disabled:opacity-50"
              >
                {saving ? 'Saving...' : 'Save Period'}
              </button>
              <button
                onClick={() => {
                  setShowAddForm(false);
                  setStartDate('');
                  setEndDate('');
                  setReason('Holiday');
                  setNotes('');
                }}
                className="border border-slate-300 text-slate-700 hover:bg-slate-50 px-4 py-2 rounded-lg font-medium"
              >
                Cancel
              </button>
            </div>
          </div>
        )}

        {/* Upcoming Periods */}
        {availability.upcomingPeriods.length > 0 && (
          <div>
            <h4 className="font-semibold text-slate-900 mb-3">Upcoming Unavailability:</h4>
            <div className="space-y-3">
              {availability.upcomingPeriods.map((period) => (
                <div
                  key={period.id}
                  className="flex items-center justify-between p-3 bg-slate-50 rounded-lg"
                >
                  <div className="flex-1">
                    <p className="font-medium text-slate-900">
                      {AvailabilityService.formatDateRange(period.startDate, period.endDate)}
                    </p>
                    {period.reason && (
                      <p className="text-sm text-slate-600">{period.reason}</p>
                    )}
                    {period.notes && (
                      <p className="text-xs text-slate-500 mt-1">{period.notes}</p>
                    )}
                  </div>
                  <button
                    onClick={() => handleDeletePeriod(period.id)}
                    className="ml-4 text-red-600 hover:text-red-700 text-sm font-medium"
                  >
                    Remove
                  </button>
                </div>
              ))}
            </div>
          </div>
        )}

        {availability.upcomingPeriods.length === 0 && !showAddForm && (
          <p className="text-slate-500 text-center py-8">
            No upcoming unavailability periods scheduled
          </p>
        )}
      </div>
    </div>
  );
};

export default MediatorAvailabilityManager;
