
import React, { useState, useEffect } from 'react';
import { Link } from 'react-router-dom';
import { AuthService } from '../../services/AuthService';
import { MediatorService } from '../../services/MediatorService';
import { BookingService } from '../../services/BookingService';
import { FDRService } from '../../services/FDRService';
import { MediatorProfilePrivate, BookingRequest, BookingStatus, FDRCase } from '../../types';

const MediatorDashboard: React.FC = () => {
  const user = AuthService.getCurrentUser();
  const [profile, setProfile] = useState<MediatorProfilePrivate | null>(null);
  const [requests, setRequests] = useState<BookingRequest[]>([]);
  const [fdrCases, setFdrCases] = useState<FDRCase[]>([]);
  const [loading, setLoading] = useState(true);

  useEffect(() => {
    if (user?.profileId) {
      loadData(user.profileId);
    }
  }, [user]);

  const loadData = async (profileId: string) => {
    const p = await MediatorService.getMediatorById(profileId, true);
    setProfile(p);
    const r = await BookingService.getByMediator(profileId);
    setRequests(r);
    const pendingFDR = await FDRService.getPendingScreenings();
    setFdrCases(pendingFDR);
    setLoading(false);
  };

  if (loading) return <div className="p-20 text-center">Loading dashboard...</div>;

  const pendingRequests = requests.filter(r => r.status === BookingStatus.DEPOSIT_PAID).length;

  return (
    <div className="max-w-7xl mx-auto px-4 py-12">
      <div className="flex justify-between items-end mb-8">
        <div>
          <h1 className="text-3xl font-bold">Mediator Dashboard</h1>
          <p className="text-slate-500">Welcome back, {profile?.displayName}</p>
        </div>
        <div className="flex gap-4">
          <Link to="/dashboard/availability" className="bg-green-100 text-green-700 px-6 py-2 rounded-lg font-bold hover:bg-green-200 transition-colors border border-green-200 flex items-center gap-2">
            📅 Availability
          </Link>
          <Link to="/dashboard/fdr-intake" className="bg-amber-100 text-amber-700 px-6 py-2 rounded-lg font-bold hover:bg-amber-200 transition-colors border border-amber-200 flex items-center gap-2">
            FDR Intake Queue
            {fdrCases.length > 0 && <span className="bg-amber-700 text-white text-[10px] px-1.5 rounded-full">{fdrCases.length}</span>}
          </Link>
          <Link to="/dashboard/profile" className="bg-indigo-600 text-white px-6 py-2 rounded-lg font-bold hover:bg-indigo-700 shadow-sm transition-colors">
            Edit Profile
          </Link>
        </div>
      </div>

      <div className="grid md:grid-cols-3 gap-6 mb-12">
        <div className="bg-white p-6 rounded-xl border border-slate-200 shadow-sm">
          <div className="text-xs font-bold text-slate-400 uppercase mb-2">Profile Status</div>
          <div className="flex items-center gap-3">
            <div className={`w-3 h-3 rounded-full ${profile?.status === 'PUBLISHED' ? 'bg-green-500' : 'bg-amber-500'}`}></div>
            <span className="text-xl font-bold text-indigo-600">{profile?.status}</span>
          </div>
        </div>

        <div className="bg-indigo-600 text-white p-6 rounded-xl shadow-lg">
          <div className="text-xs font-bold text-white/50 uppercase mb-2">New Requests</div>
          <div className="text-3xl font-bold">{pendingRequests}</div>
          <p className="text-xs text-white/50 mt-2">Awaiting your approval</p>
        </div>

        <div className="bg-white p-6 rounded-xl border border-slate-200 shadow-sm">
          <div className="text-xs font-bold text-slate-400 uppercase mb-2">FDR Tasks</div>
          <div className="text-3xl font-bold text-amber-600">{fdrCases.length}</div>
          <p className="text-xs text-slate-400 mt-2">Screenings to review</p>
        </div>
      </div>

      <div className="bg-white rounded-xl border border-slate-200 shadow-sm overflow-hidden">
        <div className="px-6 py-4 border-b border-slate-100 flex justify-between items-center text-sm">
          <h2 className="font-bold uppercase tracking-wider text-slate-400">Recent Booking Requests</h2>
          <Link to="/dashboard/requests" className="text-indigo-600 font-bold hover:underline">View All</Link>
        </div>
        <div className="divide-y divide-slate-50">
          {requests.length === 0 ? (
            <div className="p-8 text-center text-slate-400 italic">No booking requests found.</div>
          ) : (
            requests.slice(0, 5).map(req => (
              <div key={req.id} className="p-6 flex justify-between items-center hover:bg-slate-50 transition-colors">
                <div>
                  <div className="font-bold">{req.clientName}</div>
                  <div className="text-xs text-slate-500">{new Date(req.timestamp).toLocaleDateString('en-AU', { day: 'numeric', month: 'short', year: 'numeric' })}</div>
                </div>
                <span className="text-[10px] font-bold px-2 py-0.5 rounded uppercase tracking-wider bg-indigo-100 text-indigo-700">
                  {req.status}
                </span>
              </div>
            ))
          )}
        </div>
      </div>
    </div>
  );
};

export default MediatorDashboard;
