import React, { useEffect, useState } from 'react';
import { OperatingHoursService } from '../../services/OperatingHoursService';
import { OperatingHours, DayOfWeek, DaySchedule } from '../../types';

interface Props {
  mediatorId: string;
}

const OperatingHoursManager: React.FC<Props> = ({ mediatorId }) => {
  const [hours, setHours] = useState<OperatingHours | null>(null);
  const [loading, setLoading] = useState(true);
  const [saving, setSaving] = useState(false);

  useEffect(() => {
    loadHours();
  }, [mediatorId]);

  const loadHours = async () => {
    setLoading(true);
    const h = await OperatingHoursService.getOperatingHours(mediatorId);
    setHours(h);
    setLoading(false);
  };

  const handleDayToggle = async (dayOfWeek: DayOfWeek) => {
    if (!hours) return;
    
    const daySchedule = hours.schedule.find(d => d.dayOfWeek === dayOfWeek);
    if (!daySchedule) return;

    const newIsWorking = !daySchedule.isWorking;
    
    // Update local state
    const updatedSchedule = hours.schedule.map(d => 
      d.dayOfWeek === dayOfWeek 
        ? { 
            ...d, 
            isWorking: newIsWorking,
            startTime: newIsWorking && !d.startTime ? '09:00' : d.startTime,
            endTime: newIsWorking && !d.endTime ? '17:00' : d.endTime
          }
        : d
    );
    
    setHours({ ...hours, schedule: updatedSchedule });

    // Save to database
    await OperatingHoursService.updateDay(
      mediatorId,
      dayOfWeek,
      newIsWorking,
      newIsWorking ? (daySchedule.startTime || '09:00') : undefined,
      newIsWorking ? (daySchedule.endTime || '17:00') : undefined
    );
  };

  const handleTimeChange = async (dayOfWeek: DayOfWeek, field: 'startTime' | 'endTime', value: string) => {
    if (!hours) return;

    // Update local state
    const updatedSchedule = hours.schedule.map(d => 
      d.dayOfWeek === dayOfWeek ? { ...d, [field]: value } : d
    );
    
    setHours({ ...hours, schedule: updatedSchedule });

    // Debounced save
    const daySchedule = updatedSchedule.find(d => d.dayOfWeek === dayOfWeek);
    if (daySchedule) {
      await OperatingHoursService.updateDay(
        mediatorId,
        dayOfWeek,
        daySchedule.isWorking,
        daySchedule.startTime,
        daySchedule.endTime
      );
    }
  };

  const handleCopyToWeekdays = async () => {
    if (!hours) return;
    
    const monday = hours.schedule.find(d => d.dayOfWeek === DayOfWeek.MONDAY);
    if (!monday?.startTime || !monday?.endTime) {
      alert('Please set Monday hours first');
      return;
    }

    setSaving(true);
    await OperatingHoursService.copyToWeekdays(mediatorId, monday.startTime, monday.endTime);
    await loadHours();
    setSaving(false);
  };

  if (loading) {
    return (
      <div className="text-center py-8">
        <div className="inline-block w-8 h-8 border-4 border-indigo-600 border-t-transparent rounded-full animate-spin"></div>
        <p className="mt-4 text-slate-600">Loading schedule...</p>
      </div>
    );
  }

  if (!hours) return null;

  const workingDays = hours.schedule.filter(d => d.isWorking).length;

  return (
    <div className="space-y-6">
      {/* Summary Card */}
      <div className="bg-white rounded-lg shadow p-6">
        <div className="flex items-center justify-between mb-4">
          <div>
            <h2 className="text-2xl font-bold text-slate-900">Operating Hours</h2>
            <p className="text-slate-600 mt-1">
              {OperatingHoursService.getWorkingSummary(hours)}
            </p>
          </div>
          <div className="text-right">
            <div className="text-3xl font-bold text-indigo-600">{workingDays}</div>
            <div className="text-sm text-slate-500">days per week</div>
          </div>
        </div>

        <div className="bg-blue-50 border-l-4 border-blue-500 p-4">
          <p className="text-sm text-blue-900">
            <strong>💡 Tip:</strong> Set your regular working hours here. You only need to mark specific dates as unavailable 
            for holidays, illness, or other exceptions.
          </p>
        </div>
      </div>

      {/* Quick Actions */}
      <div className="bg-white rounded-lg shadow p-6">
        <h3 className="text-lg font-semibold text-slate-900 mb-4">Quick Actions</h3>
        <div className="flex gap-3">
          <button
            onClick={handleCopyToWeekdays}
            disabled={saving}
            className="bg-indigo-100 hover:bg-indigo-200 text-indigo-700 px-4 py-2 rounded-lg font-medium text-sm disabled:opacity-50"
          >
            📋 Copy Monday to All Weekdays
          </button>
        </div>
      </div>

      {/* Weekly Schedule */}
      <div className="bg-white rounded-lg shadow p-6">
        <h3 className="text-lg font-semibold text-slate-900 mb-4">Weekly Schedule</h3>
        
        <div className="space-y-3">
          {hours.schedule.map((day) => (
            <div
              key={day.dayOfWeek}
              className={`border rounded-lg p-4 transition-colors ${
                day.isWorking ? 'bg-green-50 border-green-200' : 'bg-gray-50 border-gray-200'
              }`}
            >
              <div className="flex items-center gap-4">
                {/* Day Toggle */}
                <div className="flex items-center gap-3 w-40">
                  <button
                    onClick={() => handleDayToggle(day.dayOfWeek)}
                    className={`w-12 h-6 rounded-full transition-colors relative ${
                      day.isWorking ? 'bg-green-500' : 'bg-gray-300'
                    }`}
                  >
                    <div
                      className={`w-5 h-5 bg-white rounded-full absolute top-0.5 transition-transform ${
                        day.isWorking ? 'translate-x-6' : 'translate-x-0.5'
                      }`}
                    />
                  </button>
                  <span className="font-semibold text-slate-900">
                    {OperatingHoursService.getDayName(day.dayOfWeek)}
                  </span>
                </div>

                {/* Time Pickers */}
                {day.isWorking ? (
                  <div className="flex items-center gap-3 flex-1">
                    <div className="flex items-center gap-2">
                      <label className="text-sm text-slate-600">From:</label>
                      <input
                        type="time"
                        value={day.startTime || '09:00'}
                        onChange={(e) => handleTimeChange(day.dayOfWeek, 'startTime', e.target.value)}
                        className="border border-slate-300 rounded px-3 py-1.5 text-sm"
                      />
                    </div>
                    <span className="text-slate-400">—</span>
                    <div className="flex items-center gap-2">
                      <label className="text-sm text-slate-600">To:</label>
                      <input
                        type="time"
                        value={day.endTime || '17:00'}
                        onChange={(e) => handleTimeChange(day.dayOfWeek, 'endTime', e.target.value)}
                        className="border border-slate-300 rounded px-3 py-1.5 text-sm"
                      />
                    </div>
                    <div className="text-sm text-slate-500 ml-auto">
                      {day.startTime && day.endTime && (
                        <>
                          {OperatingHoursService.formatTime(day.startTime)} - {OperatingHoursService.formatTime(day.endTime)}
                        </>
                      )}
                    </div>
                  </div>
                ) : (
                  <span className="text-slate-400 italic flex-1">Not working</span>
                )}
              </div>
            </div>
          ))}
        </div>
      </div>

      {/* Info Box */}
      <div className="bg-slate-50 rounded-lg p-6">
        <h4 className="font-semibold text-slate-900 mb-2">How This Works</h4>
        <ul className="space-y-2 text-sm text-slate-600">
          <li className="flex items-start gap-2">
            <span className="text-green-600 mt-0.5">✓</span>
            <span>Clients can only book during your operating hours</span>
          </li>
          <li className="flex items-start gap-2">
            <span className="text-green-600 mt-0.5">✓</span>
            <span>You won't appear in search results outside these times</span>
          </li>
          <li className="flex items-start gap-2">
            <span className="text-green-600 mt-0.5">✓</span>
            <span>Use "Unavailability" tab for holidays and exceptions only</span>
          </li>
          <li className="flex items-start gap-2">
            <span className="text-green-600 mt-0.5">✓</span>
            <span>Changes take effect immediately</span>
          </li>
        </ul>
      </div>
    </div>
  );
};

export default OperatingHoursManager;
