
import React, { useState, useEffect } from 'react';
import { useNavigate, Link } from 'react-router-dom';
import { AuthService } from '../../services/AuthService';
import { MediatorService } from '../../services/MediatorService';
import { SEEDED_CATEGORIES } from '../../services/SeedService';
import { FixedFeeService } from '../../services/FixedFeeService';
import { 
  MediatorProfilePrivate, 
  MediatorStatus, 
  HourlyRateBasis, 
  PricingModel 
} from '../../types';
import { dollarsToCents, centsToDollars, formatAud } from '../../lib/money';

const ProfileBuilder: React.FC = () => {
  const navigate = useNavigate();
  const user = AuthService.getCurrentUser();
  const [loading, setLoading] = useState(true);
  const [step, setStep] = useState(1);
  const [profile, setProfile] = useState<Partial<MediatorProfilePrivate>>({
    optInFixedFee: false,
    activePackages: []
  });

  useEffect(() => {
    if (user?.profileId) {
      MediatorService.getMediatorById(user.profileId, true).then(res => {
        if (res) {
          // Ensure pricing object exists
          if (!res.pricing) {
            res.pricing = {
              preMediationFeeCents: 0,
              hourlyRateCents: 0,
              hourlyRateBasis: HourlyRateBasis.TOTAL,
              travelEnabled: false,
              includedKm: 0,
              perKmCents: 0,
              accommodationEnabled: false
            };
          }
          setProfile(res);
        }
        setLoading(false);
      });
    } else {
      setLoading(false);
    }
  }, [user]);

  const handleSave = async () => {
    if (!user || !profile) return;
    
    const updatedProfile: MediatorProfilePrivate = {
      ...profile as MediatorProfilePrivate,
      userId: user.id,
      id: user.profileId || 'm-' + Math.random().toString(36).substr(2, 9),
      status: profile.status === MediatorStatus.PUBLISHED ? MediatorStatus.PUBLISHED : MediatorStatus.PENDING_APPROVAL
    };

    await MediatorService.updateProfile(updatedProfile);
    alert('Profile saved and submitted for review.');
    navigate('/dashboard');
  };

  if (loading) return <div className="p-20 text-center">Loading profile editor...</div>;

  const updatePricing = (key: keyof PricingModel, value: any) => {
    setProfile(prev => {
      const updated = {
        ...prev,
        pricing: {
          ...(prev.pricing || {
            preMediationFeeCents: 0,
            hourlyRateCents: 0,
            hourlyRateBasis: HourlyRateBasis.TOTAL,
            travelEnabled: false,
            includedKm: 0,
            perKmCents: 0,
            accommodationEnabled: false,
            hasPromotionalOffer: false,
            promotionalPreMediationFeeCents: 0,
            promotionalOfferDescription: ''
          }),
          [key]: value
        }
      };
      
      // Auto-save to database immediately
      if (user?.profileId) {
        MediatorService.updateProfile(updated as MediatorProfilePrivate).then(() => {
          console.log('✅ Pricing auto-saved:', key, value);
        });
      }
      
      return updated;
    });
  };

  const toggleCategory = (catId: string) => {
    const current = profile.categories || [];
    if (current.includes(catId)) {
      setProfile({ ...profile, categories: current.filter(id => id !== catId) });
    } else {
      setProfile({ ...profile, categories: [...current, catId] });
    }
  };

  const togglePackage = (pkgId: string) => {
    const current = profile.activePackages || [];
    if (current.includes(pkgId)) {
      setProfile({ ...profile, activePackages: current.filter(id => id !== pkgId) });
    } else {
      setProfile({ ...profile, activePackages: [...current, pkgId] });
    }
  };

  return (
    <div className="max-w-4xl mx-auto px-4 py-12">
      <div className="mb-8 flex justify-between items-center">
        <div>
          <Link to="/dashboard" className="text-indigo-600 text-sm font-bold">{"<-"} Back to Dashboard</Link>
          <h1 className="text-3xl font-bold mt-2">Mediator Profile Builder</h1>
        </div>
        <div className="flex gap-2">
          {[1, 2, 3, 4].map(s => (
            <div key={s} className={`w-3 h-3 rounded-full ${step >= s ? 'bg-indigo-600' : 'bg-slate-200'}`}></div>
          ))}
        </div>
      </div>

      <div className="bg-white rounded-2xl shadow-sm border border-slate-200 p-8">
        {step === 1 && (
          <div className="space-y-6 animate-in fade-in duration-300">
            <h2 className="text-xl font-bold border-b pb-4">Step 1: Identity & Bio</h2>
            <div className="grid md:grid-cols-2 gap-6">
              <div className="col-span-2">
                <label className="block text-sm font-semibold mb-2">Display Name</label>
                <input 
                  value={profile.displayName || ''} 
                  onChange={e => setProfile({...profile, displayName: e.target.value})}
                  className="w-full border rounded-lg p-3 outline-none focus:ring-2 focus:ring-indigo-500"
                  placeholder="e.g. Jane Smith"
                />
              </div>
              <div className="col-span-2">
                <label className="block text-sm font-semibold mb-2">Headline</label>
                <input 
                  value={profile.headline || ''} 
                  onChange={e => setProfile({...profile, headline: e.target.value})}
                  className="w-full border rounded-lg p-3 outline-none focus:ring-2 focus:ring-indigo-500"
                  placeholder="e.g. Specialist in Family and Workplace Mediation"
                />
              </div>
              <div className="col-span-2">
                <label className="block text-sm font-semibold mb-2">Biography</label>
                <textarea 
                  value={profile.bio || ''} 
                  onChange={e => setProfile({...profile, bio: e.target.value})}
                  rows={5}
                  className="w-full border rounded-lg p-3 outline-none focus:ring-2 focus:ring-indigo-500"
                  placeholder="Tell potential clients about your approach and philosophy..."
                />
              </div>
              <div className="col-span-2">
                <label className="block text-sm font-semibold mb-2">Photo URL</label>
                <input 
                  value={profile.photoUrl || ''} 
                  onChange={e => setProfile({...profile, photoUrl: e.target.value})}
                  className="w-full border rounded-lg p-3 outline-none focus:ring-2 focus:ring-indigo-500"
                  placeholder="https://example.com/photo.jpg"
                />
              </div>
              <div className="col-span-2">
                <label className="block text-sm font-semibold mb-2">Forward-To Email (Private)</label>
                <input 
                  value={profile.forwardToEmail || ''} 
                  onChange={e => setProfile({...profile, forwardToEmail: e.target.value})}
                  className="w-full border rounded-lg p-3 outline-none focus:ring-2 focus:ring-indigo-500"
                  placeholder="e.g. yourname@gmail.com"
                />
              </div>
            </div>
            <div className="flex justify-end pt-6">
              <button onClick={() => setStep(2)} className="bg-indigo-600 text-white px-8 py-3 rounded-xl font-bold hover:bg-indigo-700 transition-colors">
                Next: Expertise {"->"}
              </button>
            </div>
          </div>
        )}

        {step === 2 && (
          <div className="space-y-6 animate-in fade-in duration-300">
            <h2 className="text-xl font-bold border-b pb-4">Step 2: Expertise & Locations</h2>
            <div>
              <label className="block text-sm font-semibold mb-4">Mediation Categories Experienced In</label>
              <div className="grid grid-cols-2 md:grid-cols-3 gap-3">
                {SEEDED_CATEGORIES.map(cat => (
                  <button
                    key={cat.id}
                    onClick={() => toggleCategory(cat.id)}
                    className={`text-left p-3 rounded-lg border text-sm transition-all ${
                      profile.categories?.includes(cat.id) 
                      ? 'bg-indigo-50 border-indigo-600 text-indigo-700 font-bold' 
                      : 'border-slate-200 text-slate-600 hover:border-indigo-300'
                    }`}
                  >
                    {cat.name}
                  </button>
                ))}
              </div>
            </div>
            <div>
              <label className="block text-sm font-semibold mb-2">Locations (Comma separated)</label>
              <input 
                value={profile.locations?.join(', ') || ''} 
                onChange={e => setProfile({...profile, locations: e.target.value.split(',').map(s => s.trim())})}
                className="w-full border rounded-lg p-3 outline-none focus:ring-2 focus:ring-indigo-500"
                placeholder="e.g. Sydney, Melbourne, Brisbane"
              />
            </div>
            <label className="flex items-center gap-3 cursor-pointer p-4 bg-slate-50 rounded-xl">
              <input 
                type="checkbox" 
                checked={profile.onlineAvailable || false}
                onChange={e => setProfile({...profile, onlineAvailable: e.target.checked})}
                className="w-5 h-5 text-indigo-600"
              />
              <span className="font-semibold text-slate-700">I am available for online/video mediation</span>
            </label>
            <div className="flex justify-between pt-6">
              <button onClick={() => setStep(1)} className="text-slate-500 font-bold px-8 py-3">{"<-"} Back</button>
              <button onClick={() => setStep(3)} className="bg-indigo-600 text-white px-8 py-3 rounded-xl font-bold hover:bg-indigo-700">
                Next: Pricing {"->"}
              </button>
            </div>
          </div>
        )}

        {step === 3 && (
          <div className="space-y-6 animate-in fade-in duration-300">
            <h2 className="text-xl font-bold border-b pb-4">Step 3: Private Pricing Policy</h2>
            <div className="bg-amber-50 border border-amber-200 p-4 rounded-xl text-amber-800 text-sm mb-6">
              <strong>GST Compliance:</strong> All pricing entered must be inclusive of GST.
            </div>
            <div className="grid md:grid-cols-2 gap-8">
              <div className="space-y-4">
                <h3 className="font-bold text-slate-400 text-xs uppercase tracking-widest">Base Rates</h3>
                <div>
                  <label className="block text-sm font-semibold mb-1">Pre-Mediation Fee ($ AUD Inc. GST)</label>
                  <input 
                    type="number"
                    key={`premedkey-${profile.pricing?.preMediationFeeCents}`}
                    defaultValue={centsToDollars(profile.pricing?.preMediationFeeCents || 0)} 
                    onBlur={e => updatePricing('preMediationFeeCents', dollarsToCents(Number(e.target.value)))}
                    className="w-full border rounded-lg p-3 outline-none focus:ring-2 focus:ring-indigo-500"
                  />
                </div>
                <div>
                  <label className="block text-sm font-semibold mb-1">Hourly Rate ($ AUD Inc. GST)</label>
                  <input 
                    type="number"
                    key={`hourlykey-${profile.pricing?.hourlyRateCents}`}
                    defaultValue={centsToDollars(profile.pricing?.hourlyRateCents || 0)} 
                    onBlur={e => updatePricing('hourlyRateCents', dollarsToCents(Number(e.target.value)))}
                    className="w-full border rounded-lg p-3 outline-none focus:ring-2 focus:ring-indigo-500"
                  />
                </div>
              </div>

              {/* Promotional Offers Section */}
              <div className="border-t pt-6 mt-6">
                <h3 className="font-bold text-slate-400 text-xs uppercase tracking-widest mb-4">Promotional Offers (Optional)</h3>
                
                <div className="bg-blue-50 border-l-4 border-blue-500 p-4 mb-4">
                  <p className="text-sm text-blue-900">
                    <strong>💡 Attract new clients</strong> with a discounted initial consultation. 
                    This promotional rate will be available to <strong>ALL parties</strong> in any mediation to ensure equal treatment.
                  </p>
                </div>

                <label className="flex items-start gap-3 cursor-pointer mb-4">
                  <input 
                    type="checkbox" 
                    checked={profile.pricing?.hasPromotionalOffer || false}
                    onChange={e => updatePricing('hasPromotionalOffer', e.target.checked)}
                    className="mt-1 w-5 h-5 text-indigo-600 rounded"
                  />
                  <div>
                    <span className="font-semibold text-slate-900 block">Offer promotional rate</span>
                    <span className="text-xs text-slate-600">Enable a special introductory rate for initial consultations</span>
                  </div>
                </label>

                {profile.pricing?.hasPromotionalOffer && (
                  <div className="space-y-4 ml-8 p-4 bg-gray-50 rounded-lg">
                    <div>
                      <label className="block text-sm font-semibold mb-1">
                        Promotional Pre-Mediation Fee ($ AUD Inc. GST)
                      </label>
                      <input 
                        type="number"
                        key={`promokey-${profile.pricing?.promotionalPreMediationFeeCents}`}
                        defaultValue={centsToDollars(profile.pricing?.promotionalPreMediationFeeCents || 0)} 
                        onBlur={e => updatePricing('promotionalPreMediationFeeCents', dollarsToCents(Number(e.target.value)))}
                        className="w-full border rounded-lg p-3 outline-none focus:ring-2 focus:ring-indigo-500"
                        placeholder="e.g., 175"
                      />
                      {profile.pricing?.preMediationFeeCents && profile.pricing?.promotionalPreMediationFeeCents && (
                        <p className="text-sm text-green-700 mt-2">
                          ✓ Saving: ${centsToDollars(profile.pricing.preMediationFeeCents - profile.pricing.promotionalPreMediationFeeCents)} 
                          ({Math.round((1 - profile.pricing.promotionalPreMediationFeeCents / profile.pricing.preMediationFeeCents) * 100)}% discount)
                        </p>
                      )}
                    </div>

                    <div>
                      <label className="block text-sm font-semibold mb-1">
                        Offer Description (Optional)
                      </label>
                      <input 
                        type="text"
                        key={`desckey-${profile.pricing?.promotionalOfferDescription}`}
                        defaultValue={profile.pricing?.promotionalOfferDescription || ''} 
                        onBlur={e => updatePricing('promotionalOfferDescription', e.target.value)}
                        className="w-full border rounded-lg p-3 outline-none focus:ring-2 focus:ring-indigo-500"
                        placeholder="e.g., First consultation special rate"
                        maxLength={100}
                      />
                      <p className="text-xs text-slate-500 mt-1">This will be shown to clients on your profile</p>
                    </div>

                    <div className="bg-amber-50 border-l-4 border-amber-500 p-4">
                      <label className="flex items-start gap-3 cursor-pointer">
                        <input 
                          type="checkbox" 
                          required
                          className="mt-1 w-4 h-4 text-amber-600 rounded"
                        />
                        <div className="text-sm text-amber-900">
                          <strong>Ethical Confirmation Required:</strong> I understand that this promotional rate must be 
                          offered equally to ALL parties in any mediation to maintain impartiality and comply with mediator 
                          ethical standards. The platform will automatically apply this rate to all parties in the same case.
                        </div>
                      </label>
                    </div>
                  </div>
                )}
              </div>
            </div>
            <div className="flex justify-between pt-6">
              <button onClick={() => setStep(2)} className="text-slate-500 font-bold px-8 py-3 hover:text-indigo-600 transition-colors">{"<-"} Back</button>
              <button onClick={() => setStep(4)} className="bg-indigo-600 text-white px-8 py-3 rounded-xl font-bold hover:bg-indigo-700">
                Next: Subscriptions {"->"}
              </button>
            </div>
          </div>
        )}

        {step === 4 && (
          <div className="space-y-8 animate-in fade-in duration-300">
            <div>
              <h2 className="text-xl font-bold border-b pb-4">Step 4: Fixed Fee Subscription</h2>
              <p className="text-slate-500 mt-4 text-sm leading-relaxed">
                Opt-in to the <strong>Fixed Fee Program</strong> to offer standardized online mediation packages. This makes your services easier to book and streamlines the payment process.
              </p>
            </div>

            <label className="flex items-center gap-4 cursor-pointer p-6 bg-indigo-50 rounded-2xl border-2 border-indigo-200">
              <input 
                type="checkbox" 
                checked={profile.optInFixedFee}
                onChange={e => setProfile({...profile, optInFixedFee: e.target.checked})}
                className="w-6 h-6 text-indigo-600 rounded"
              />
              <div>
                <span className="font-bold text-indigo-900 block">Join Fixed Fee Program</span>
                <span className="text-xs text-indigo-700/70">I agree to adopt the fixed fee approach for standardized online mediations.</span>
              </div>
            </label>

            {profile.optInFixedFee && (
              <div className="space-y-6 animate-in slide-in-from-top-4">
                <h3 className="font-bold text-slate-700">Select Qualified Packages</h3>
                <div className="grid gap-4">
                  {FixedFeeService.getPackages().map(pkg => (
                    <div 
                      key={pkg.id}
                      onClick={() => togglePackage(pkg.id)}
                      className={`p-4 rounded-xl border-2 transition-all cursor-pointer flex justify-between items-center ${
                        profile.activePackages?.includes(pkg.id)
                        ? 'bg-white border-indigo-600 shadow-md ring-4 ring-indigo-50'
                        : 'bg-slate-50 border-slate-200 opacity-60'
                      }`}
                    >
                      <div>
                        <div className="font-bold text-indigo-900">{pkg.name}</div>
                        <div className="text-xs text-slate-500">{pkg.description}</div>
                      </div>
                      <div className="text-right">
                        <div className="font-bold text-indigo-600">{formatAud(pkg.totalFeeCents)}</div>
                        <div className="text-[10px] text-slate-400 uppercase font-bold">{pkg.durationHours} Hours</div>
                      </div>
                    </div>
                  ))}
                </div>
              </div>
            )}

            <div className="flex justify-between pt-6">
              <button onClick={() => setStep(3)} className="text-slate-500 font-bold px-8 py-3">{"<-"} Back</button>
              <button onClick={handleSave} className="bg-green-600 text-white px-8 py-3 rounded-xl font-bold hover:bg-green-700 shadow-lg">
                Finish & Submit Review
              </button>
            </div>
          </div>
        )}
      </div>
    </div>
  );
};

export default ProfileBuilder;
