
import React, { useState, useEffect } from 'react';
import { Link } from 'react-router-dom';
import { AuthService } from '../../services/AuthService';
import { BookingService } from '../../services/BookingService';
import { FixedFeeService } from '../../services/FixedFeeService';
import { BookingRequest, BookingStatus, PaymentKind } from '../../types';
import { formatAud, dollarsToCents } from '../../lib/money';

const RequestManager: React.FC = () => {
  const user = AuthService.getCurrentUser();
  const [requests, setRequests] = useState<BookingRequest[]>([]);
  const [loading, setLoading] = useState(true);
  const [processingId, setProcessingId] = useState<string | null>(null);
  const [totalFeeInput, setTotalFeeInput] = useState<string>('');

  useEffect(() => {
    if (user?.profileId) {
      loadRequests(user.profileId);
    }
  }, [user]);

  const loadRequests = async (profileId: string) => {
    const data = await BookingService.getByMediator(profileId);
    setRequests(data.sort((a, b) => b.timestamp - a.timestamp));
    setLoading(false);
  };

  const handleProceed = async (req: BookingRequest) => {
    const fee = req.isFixedFee 
      ? (FixedFeeService.getPackageById(req.packageId!)?.totalFeeCents || 0)
      : dollarsToCents(Number(totalFeeInput));
      
    if (!fee && !req.isFixedFee) return alert('Please enter total fee');
    
    await BookingService.updateStatus(req.id, BookingStatus.PROCEED_APPROVED, fee);
    setTotalFeeInput('');
    setProcessingId(null);
    if (user?.profileId) loadRequests(user.profileId);
  };

  const handleSimulateBalance = async (req: BookingRequest) => {
    const balance = (req.feeTotalCents || 0) - req.depositAmountCents;
    await BookingService.recordPayment({
      id: 'pay-' + Math.random().toString(36).substr(2, 9),
      bookingId: req.id,
      kind: PaymentKind.BALANCE,
      amountCents: balance,
      timestamp: Date.now()
    });
    if (user?.profileId) loadRequests(user.profileId);
  };

  if (loading) return <div className="p-20 text-center">Loading requests...</div>;

  return (
    <div className="max-w-7xl mx-auto px-4 py-12">
      <div className="mb-8">
        <Link to="/dashboard" className="text-indigo-600 text-sm font-bold flex items-center gap-1 mb-2">{"<-"} Back to Dashboard</Link>
        <h1 className="text-3xl font-bold">Manage Booking Requests</h1>
      </div>

      <div className="space-y-6">
        {requests.length === 0 ? (
          <div className="p-20 text-center bg-white rounded-xl border border-slate-200 text-slate-400 italic">No requests yet.</div>
        ) : (
          requests.map(req => {
            const pkg = req.packageId ? FixedFeeService.getPackageById(req.packageId) : null;
            return (
              <div key={req.id} className={`bg-white rounded-xl shadow-sm border overflow-hidden ${req.isFixedFee ? 'border-indigo-200 ring-2 ring-indigo-50' : 'border-slate-200'}`}>
                <div className="p-6 md:flex justify-between items-start gap-8">
                  <div className="flex-grow space-y-4">
                    <div className="flex items-center gap-4">
                      <h3 className="text-xl font-bold">{req.clientName}</h3>
                      <span className={`text-[10px] font-bold px-2 py-0.5 rounded uppercase tracking-wider ${
                        req.status === 'DEPOSIT_PAID' ? 'bg-indigo-100 text-indigo-700' : 
                        req.status === 'PROCEED_APPROVED' ? 'bg-amber-100 text-amber-700' : 'bg-green-100 text-green-700'
                      }`}>
                        {req.status}
                      </span>
                      {req.isFixedFee && (
                        <span className="bg-indigo-600 text-white text-[9px] font-bold px-2 py-0.5 rounded uppercase">Fixed Fee</span>
                      )}
                    </div>
                    
                    <div className="grid md:grid-cols-2 gap-4 text-sm text-slate-600">
                      <div>
                        <span className="font-semibold block">Contact:</span>
                        {req.clientEmail}
                      </div>
                      <div>
                        <span className="font-semibold block">Proposed Session:</span>
                        <span className="text-indigo-700 font-bold">{req.proposedDateTime ? new Date(req.proposedDateTime).toLocaleString() : 'Not set'}</span>
                      </div>
                      <div className="md:col-span-2">
                        <span className="font-semibold block">{req.isFixedFee ? 'Standardized Package:' : 'Client Notes:'}</span>
                        <p className="bg-slate-50 p-3 rounded mt-1 italic">
                          {req.isFixedFee ? (
                            <span><strong>{pkg?.name}</strong> - {pkg?.durationHours}hr Session @ {formatAud(pkg?.totalFeeCents || 0)}</span>
                          ) : `"${req.notes}"`}
                        </p>
                      </div>
                    </div>
                  </div>

                  <div className="mt-6 md:mt-0 w-full md:w-64 border-t md:border-t-0 md:border-l border-slate-100 pt-6 md:pt-0 md:pl-6 space-y-4">
                    <div className="text-xs font-bold text-slate-400 uppercase">Financial Summary</div>
                    <div className="space-y-2 text-sm">
                      <div className="flex justify-between text-slate-600"><span>Deposit Received:</span> <span>{formatAud(req.depositAmountCents)}</span></div>
                      {req.feeTotalCents && <div className="flex justify-between font-bold text-slate-900"><span>Confirmed Fee:</span> <span>{formatAud(req.feeTotalCents)}</span></div>}
                    </div>

                    {req.status === BookingStatus.DEPOSIT_PAID && (
                      req.isFixedFee ? (
                        <button 
                          onClick={() => handleProceed(req)}
                          className="w-full bg-indigo-600 hover:bg-indigo-700 text-white font-bold py-3 rounded text-sm transition-colors shadow-lg"
                        >
                          Confirm Fixed Quote
                        </button>
                      ) : (
                        processingId === req.id ? (
                          <div className="space-y-3 pt-4 border-t border-slate-100">
                            <label className="text-xs font-bold block">Final Confirmed Fee ($ AUD)</label>
                            <input 
                              type="number" 
                              value={totalFeeInput}
                              onChange={(e) => setTotalFeeInput(e.target.value)}
                              placeholder="e.g. 1500"
                              className="w-full border rounded p-2 text-sm"
                            />
                            <div className="flex gap-2">
                              <button onClick={() => handleProceed(req)} className="flex-grow bg-indigo-600 text-white text-xs font-bold py-2 rounded">Approve</button>
                              <button onClick={() => setProcessingId(null)} className="bg-slate-100 text-slate-600 px-3 py-2 rounded text-xs">✕</button>
                            </div>
                          </div>
                        ) : (
                          <button 
                            onClick={() => setProcessingId(req.id)}
                            className="w-full bg-indigo-600 hover:bg-indigo-700 text-white font-bold py-3 rounded text-sm transition-colors"
                          >
                            Accept & Quote
                          </button>
                        )
                      )
                    )}

                    {req.status === BookingStatus.PROCEED_APPROVED && (
                      <button 
                        onClick={() => handleSimulateBalance(req)}
                        className="w-full bg-green-600 hover:bg-green-700 text-white font-bold py-3 rounded text-sm transition-colors"
                      >
                        Simulate Balance Payment
                      </button>
                    )}
                  </div>
                </div>
              </div>
            );
          })
        )}
      </div>
    </div>
  );
};

export default RequestManager;
