
import React, { useState, useEffect } from 'react';
import { useParams, Link } from 'react-router-dom';
import { BookingRequest, BookingStatus, PaymentEvent, MediatorProfilePublic } from '../../types';
import { BookingService } from '../../services/BookingService';
import { MediatorService } from '../../services/MediatorService';
import { formatAud } from '../../lib/money';

const BookingStatusPage: React.FC = () => {
  const { id } = useParams<{ id: string }>();
  const [booking, setBooking] = useState<BookingRequest | null>(null);
  const [mediator, setMediator] = useState<MediatorProfilePublic | null>(null);
  const [payments, setPayments] = useState<PaymentEvent[]>([]);
  const [loading, setLoading] = useState(true);

  useEffect(() => {
    if (id) {
      loadData(id);
    }
  }, [id]);

  const loadData = async (bookingId: string) => {
    const b = await BookingService.getById(bookingId);
    if (b) {
      setBooking(b);
      const m = await MediatorService.getMediatorById(b.mediatorId);
      setMediator(m);
      const p = await BookingService.getPaymentsForBooking(bookingId);
      setPayments(p);
    }
    setLoading(false);
  };

  if (loading) return <div className="p-20 text-center text-slate-500">Retrieving booking status...</div>;
  if (!booking) return <div className="p-20 text-center text-red-500">Booking reference not found.</div>;

  const statuses = [
    { key: BookingStatus.PENDING, label: 'Submitted' },
    { key: BookingStatus.DEPOSIT_PAID, label: 'Deposit Paid' },
    { key: BookingStatus.PROCEED_APPROVED, label: 'Approved to Proceed' },
    { key: BookingStatus.BALANCE_PAID, label: 'Confirmed & Settled' }
  ];

  const currentIndex = statuses.findIndex(s => s.key === booking.status);

  return (
    <div className="max-w-4xl mx-auto px-4 py-12">
      <div className="bg-white rounded-2xl shadow-xl overflow-hidden border border-slate-100">
        <div className="bg-indigo-600 px-8 py-10 text-white">
          <div className="flex justify-between items-start mb-6">
            <div>
              <div className="text-white/50 text-xs font-bold uppercase tracking-widest mb-2">Booking Reference</div>
              <h1 className="text-3xl font-bold">{booking.id}</h1>
            </div>
            <div className="text-right">
              <div className="text-white/50 text-xs font-bold uppercase mb-2">Current Status</div>
              <span className="bg-white/20 px-3 py-1 rounded-full text-sm font-bold border border-white/10">{booking.status}</span>
            </div>
          </div>
          
          <div className="grid md:grid-cols-2 gap-8 text-sm">
            <div>
              <div className="text-white/50 mb-1 text-[10px] font-bold uppercase tracking-wider">Mediator</div>
              <div className="font-bold text-lg mb-0.5">{mediator?.displayName}</div>
              {mediator?.publicEmail && (
                <div className="text-white/70 underline text-sm">{mediator.publicEmail}</div>
              )}
            </div>
            <div className="text-right md:text-left">
              <div className="text-white/50 mb-1 text-[10px] font-bold uppercase tracking-wider">Requested By</div>
              <div className="font-bold text-lg">{booking.clientName}</div>
              <div className="text-white/70">{booking.clientEmail}</div>
            </div>
          </div>
        </div>

        <div className="p-8">
          <h2 className="text-xl font-bold mb-10">Booking Timeline</h2>
          <div className="relative">
            <div className="absolute left-4 top-0 bottom-0 w-0.5 bg-slate-100"></div>
            <div className="space-y-12">
              {statuses.map((s, i) => {
                const isCompleted = i <= currentIndex;
                const isCurrent = i === currentIndex;
                return (
                  <div key={s.key} className="relative pl-12">
                    <div className={`absolute left-1.5 w-6 h-6 rounded-full border-4 flex items-center justify-center -translate-x-1/2 z-10 ${
                      isCompleted ? 'bg-indigo-600 border-indigo-500' : 'bg-white border-slate-100'
                    }`}>
                      {isCompleted && (
                        <svg className="h-2 w-2 text-white" fill="currentColor" viewBox="0 0 20 20">
                          <path fillRule="evenodd" d="M16.707 5.293a1 1 0 010 1.414l-8 8a1 1 0 01-1.414 0l-4-4a1 1 0 011.414-1.414L8 12.586l7.293-7.293a1 1 0 011.414 0z" clipRule="evenodd" />
                        </svg>
                      )}
                    </div>
                    <div>
                      <h3 className={`font-bold ${isCurrent ? 'text-indigo-600 text-lg' : isCompleted ? 'text-slate-900' : 'text-slate-300'}`}>
                        {s.label}
                      </h3>
                      {isCurrent && (
                        <div className="mt-2 text-sm text-slate-600 bg-slate-50 p-4 rounded-xl border border-slate-100 animate-pulse">
                          {booking.status === BookingStatus.DEPOSIT_PAID && "Mediator notified. Reviewing case details..."}
                          {booking.status === BookingStatus.PROCEED_APPROVED && `Approved! Settle balance of ${formatAud((booking.feeTotalCents || 0) - booking.depositAmountCents)} to confirm.`}
                          {booking.status === BookingStatus.BALANCE_PAID && "Mediation confirmed. Mediator will contact you shortly."}
                        </div>
                      )}
                    </div>
                  </div>
                );
              })}
            </div>
          </div>
        </div>

        <div className="bg-slate-50 p-8 border-t border-slate-100">
          <h2 className="text-sm font-bold text-slate-400 uppercase tracking-widest mb-4">Payment History</h2>
          <div className="space-y-3">
            {payments.map(p => (
              <div key={p.id} className="flex justify-between items-center bg-white p-4 rounded-lg shadow-sm border border-slate-200">
                <div>
                  <div className="font-bold text-slate-800">{p.kind === 'DEPOSIT' ? 'Deposit' : 'Balance'}</div>
                  <div className="text-xs text-slate-400">{new Date(p.timestamp).toLocaleDateString('en-AU', { day: 'numeric', month: 'short', year: 'numeric' })}</div>
                </div>
                <div className="text-green-600 font-bold">+{formatAud(p.amountCents)}</div>
              </div>
            ))}
          </div>
        </div>
      </div>
    </div>
  );
};

export default BookingStatusPage;
