import React, { useEffect, useState } from 'react';
import { useNavigate, useSearchParams } from 'react-router-dom';

const BookingSuccessPage: React.FC = () => {
  const [searchParams] = useSearchParams();
  const navigate = useNavigate();
  const [loading, setLoading] = useState(true);
  const sessionId = searchParams.get('session_id');

  useEffect(() => {
    const verifyPayment = async () => {
      if (!sessionId) {
        navigate('/');
        return;
      }

      try {
        const apiUrl = import.meta.env.VITE_PAYMENT_API_URL || 'http://localhost:3001';
        const response = await fetch(`${apiUrl}/api/verify-payment`, {
          method: 'POST',
          headers: { 'Content-Type': 'application/json' },
          body: JSON.stringify({ sessionId })
        });

        if (response.ok) {
          const data = await response.json();
          console.log('✅ Payment verified, emails sent:', data);
        }
      } catch (error) {
        console.error('Payment verification error:', error);
      } finally {
        setLoading(false);
      }
    };

    verifyPayment();
  }, [sessionId, navigate]);

  if (loading) {
    return (
      <div className="min-h-screen bg-gradient-to-br from-indigo-50 via-white to-purple-50 flex items-center justify-center">
        <div className="text-center">
          <div className="inline-block w-12 h-12 border-4 border-indigo-600 border-t-transparent rounded-full animate-spin"></div>
          <p className="mt-4 text-slate-600">Processing your payment...</p>
        </div>
      </div>
    );
  }

  return (
    <div className="min-h-screen bg-gradient-to-br from-indigo-50 via-white to-purple-50 flex items-center justify-center p-4">
      <div className="max-w-2xl w-full bg-white rounded-2xl shadow-xl p-8 md:p-12">
        {/* Success Icon */}
        <div className="text-center mb-8">
          <div className="inline-flex items-center justify-center w-20 h-20 bg-green-100 rounded-full mb-6">
            <svg className="w-10 h-10 text-green-600" fill="none" stroke="currentColor" viewBox="0 0 24 24">
              <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M5 13l4 4L19 7" />
            </svg>
          </div>
          <h1 className="text-3xl md:text-4xl font-bold text-slate-900 mb-4">
            Payment Successful! 🎉
          </h1>
          <p className="text-lg text-slate-600">
            Your booking has been confirmed
          </p>
        </div>

        {/* What Happens Next */}
        <div className="bg-indigo-50 rounded-xl p-6 mb-8">
          <h2 className="text-xl font-bold text-slate-900 mb-4">
            What happens next:
          </h2>
          <ol className="space-y-3">
            <li className="flex items-start">
              <span className="flex items-center justify-center w-6 h-6 bg-indigo-600 text-white rounded-full text-sm font-bold mr-3 mt-0.5">1</span>
              <span className="text-slate-700">You'll receive a confirmation email with your booking reference</span>
            </li>
            <li className="flex items-start">
              <span className="flex items-center justify-center w-6 h-6 bg-indigo-600 text-white rounded-full text-sm font-bold mr-3 mt-0.5">2</span>
              <span className="text-slate-700">Your mediator will contact you within 48 hours</span>
            </li>
            <li className="flex items-start">
              <span className="flex items-center justify-center w-6 h-6 bg-indigo-600 text-white rounded-full text-sm font-bold mr-3 mt-0.5">3</span>
              <span className="text-slate-700">They'll send you the Agreement to Mediate to review</span>
            </li>
            <li className="flex items-start">
              <span className="flex items-center justify-center w-6 h-6 bg-indigo-600 text-white rounded-full text-sm font-bold mr-3 mt-0.5">4</span>
              <span className="text-slate-700">You'll schedule your pre-mediation session together</span>
            </li>
          </ol>
        </div>

        {/* Payment Details */}
        <div className="border-t border-slate-200 pt-6 mb-8">
          <p className="text-sm text-slate-500">
            Payment Reference: <span className="font-mono text-slate-700">{sessionId?.slice(-12)}</span>
          </p>
        </div>

        {/* Actions */}
        <div className="flex flex-col sm:flex-row gap-4">
          <button
            onClick={() => navigate('/')}
            className="flex-1 bg-indigo-600 hover:bg-indigo-700 text-white font-semibold py-3 px-6 rounded-lg transition-colors"
          >
            Return to Home
          </button>
          <button
            onClick={() => navigate('/mediators')}
            className="flex-1 border-2 border-indigo-600 text-indigo-600 hover:bg-indigo-50 font-semibold py-3 px-6 rounded-lg transition-colors"
          >
            Browse Mediators
          </button>
        </div>

        {/* Support */}
        <div className="mt-8 text-center text-sm text-slate-500">
          <p>
            Questions? Contact us at{' '}
            <a href="mailto:support@medi8.com.au" className="text-indigo-600 hover:text-indigo-700 underline">
              support@medi8.com.au
            </a>
          </p>
        </div>
      </div>
    </div>
  );
};

export default BookingSuccessPage;
