import React, { useState, useEffect } from 'react';
import { useNavigate } from 'react-router-dom';
import {
  EngagementForm as EngagementFormType,
  EngagementStage,
  DisputeType,
  MediatorProfilePublic
} from '../../types';
import { EngagementService } from '../../services/EngagementService';
import { BookingService } from '../../services/BookingService';
import { EmailService } from '../../services/EmailService';
import { StripeService } from '../../services/StripeService';
import { formatAud } from '../../lib/money';

interface Props {
  mediator: MediatorProfilePublic;
  estimatedFeeCents: number;
  onBack: () => void;
}

const EngagementFormComponent: React.FC<Props> = ({ mediator, estimatedFeeCents, onBack }) => {
  const navigate = useNavigate();
  const [formData, setFormData] = useState<EngagementFormType | null>(null);
  const [currentStep, setCurrentStep] = useState(1);
  const [saving, setSaving] = useState(false);
  const [submitting, setSubmitting] = useState(false);

  // Initialize form on mount
  useEffect(() => {
    const initForm = async () => {
      const form = await EngagementService.createForm(mediator.id, estimatedFeeCents);
      setFormData(form);
    };
    initForm();
  }, [mediator.id, estimatedFeeCents]);

  // Auto-save whenever formData changes
  useEffect(() => {
    if (!formData) return;
    
    const autoSave = async () => {
      setSaving(true);
      await EngagementService.updateForm(formData.id, formData);
      setSaving(false);
    };

    const timer = setTimeout(autoSave, 1000); // Debounce 1 second
    return () => clearTimeout(timer);
  }, [formData]);

  const updateField = <K extends keyof EngagementFormType>(field: K, value: EngagementFormType[K]) => {
    if (formData) {
      setFormData({ ...formData, [field]: value });
    }
  };

  const canProceed = (step: number): boolean => {
    if (!formData) return false;
    
    switch (step) {
      case 1: // Contact Info
        return !!(formData.clientName && formData.clientEmail && formData.clientPhone && formData.preferredLocation);
      case 2: // Dispute Details
        return !!(formData.disputeType && formData.disputeDuration && formData.otherPartyName && formData.disputeDescription);
      case 3: // Legal Context
        return true; // All fields optional
      case 4: // Review
        return true;
      default:
        return false;
    }
  };

  const nextStep = () => {
    if (canProceed(currentStep)) {
      setCurrentStep(prev => Math.min(prev + 1, 4));
    }
  };

  const prevStep = () => {
    setCurrentStep(prev => Math.max(prev - 1, 1));
  };

  const handleProceedToPayment = async () => {
    if (!formData) return;
    
    setSubmitting(true);
    
    try {
      // Mark form as complete
      await EngagementService.completeForm(formData.id);
      
      // Create booking ID for tracking
      const bookingId = 'bk-' + Math.random().toString(36).substr(2, 9);
      
      // Create booking request (pending payment)
      await BookingService.createRequest({
        id: bookingId,
        mediatorId: mediator.id,
        clientName: formData.clientName,
        clientEmail: formData.clientEmail,
        notes: `Dispute Type: ${formData.disputeType.replace(/_/g, ' ')}\n\n${formData.disputeDescription}\n\nLegal Context:\n- Court orders: ${formData.hasCourtOrders ? 'Yes' : 'No'}\n- Safety orders: ${formData.hasSafetyOrders ? 'Yes' : 'No'}\n- Contracts: ${formData.hasContracts ? 'Yes' : 'No'}\n${formData.contextDetails ? '\nDetails: ' + formData.contextDetails : ''}\n\nSpecial Considerations:\n- Disability accommodation: ${formData.needsDisabilityAccommodation ? 'Required' : 'Not required'}\n- Cultural/Language: ${formData.culturalOrLanguageNeeds || 'None specified'}\n- Safety concerns: ${formData.safetyConcerns || 'None specified'}`,
        status: 'PENDING' as any,
        depositAmountCents: estimatedFeeCents,
        timestamp: Date.now(),
        isFixedFee: false,
        proposedDateTime: Date.now() + (7 * 24 * 60 * 60 * 1000),
        esignStatus: 'NOT_SENT' as any
      });
      
      // Create Stripe Checkout session and redirect
      await StripeService.createCheckoutSession({
        mediatorId: mediator.id,
        mediatorName: mediator.displayName,
        clientName: formData.clientName,
        clientEmail: formData.clientEmail,
        amount: estimatedFeeCents,
        description: 'Pre-Mediation Session',
        metadata: {
          bookingId,
          formId: formData.id,
          disputeType: formData.disputeType
        }
      });
      
      // Note: After successful payment, Stripe will redirect to success page
      // The success page should handle sending confirmation emails
      
    } catch (error) {
      console.error('Payment error:', error);
      alert('Payment processing failed. Please try again.');
      setSubmitting(false);
    }
  };

  if (!formData) {
    return (
      <div className="text-center py-8">
        <div className="inline-block w-8 h-8 border-4 border-indigo-600 border-t-transparent rounded-full animate-spin"></div>
        <p className="mt-4 text-slate-600">Preparing your intake form...</p>
      </div>
    );
  }

  return (
    <div className="space-y-6 animate-in fade-in slide-in-from-right-4 duration-300">
      {/* Progress Indicator */}
      <div className="flex items-center justify-between mb-8">
        {[1, 2, 3, 4].map(step => (
          <div key={step} className="flex items-center flex-1">
            <div className={`flex items-center justify-center w-10 h-10 rounded-full font-bold ${
              currentStep >= step ? 'bg-indigo-600 text-white' : 'bg-slate-200 text-slate-500'
            }`}>
              {step}
            </div>
            {step < 4 && (
              <div className={`flex-1 h-1 mx-2 ${currentStep > step ? 'bg-indigo-600' : 'bg-slate-200'}`} />
            )}
          </div>
        ))}
      </div>

      {/* Auto-save indicator */}
      {saving && (
        <div className="text-xs text-indigo-600 flex items-center gap-2">
          <svg className="animate-spin h-3 w-3" viewBox="0 0 24 24">
            <circle className="opacity-25" cx="12" cy="12" r="10" stroke="currentColor" strokeWidth="4" fill="none" />
            <path className="opacity-75" fill="currentColor" d="M4 12a8 8 0 018-8V0C5.373 0 0 5.373 0 12h4zm2 5.291A7.962 7.962 0 014 12H0c0 3.042 1.135 5.824 3 7.938l3-2.647z" />
          </svg>
          Saving...
        </div>
      )}

      {/* Step 1: Contact Information */}
      {currentStep === 1 && (
        <div className="space-y-6">
          <div>
            <button onClick={onBack} className="text-white/70 hover:text-white text-sm flex items-center gap-1 mb-4">
              ← Back to profile
            </button>
            <h2 className="text-2xl font-bold mb-2">Contact Information</h2>
            <p className="text-white/70 text-sm">Tell us how to reach you</p>
          </div>

          <div className="space-y-4">
            <div>
              <label className="block text-[10px] font-bold text-white/50 uppercase mb-2">Full Name *</label>
              <input
                type="text"
                value={formData.clientName}
                onChange={(e) => updateField('clientName', e.target.value)}
                placeholder="Jane Doe"
                className="w-full bg-white/10 border-white/20 rounded-lg p-3 text-white placeholder-white/30 focus:ring-2 focus:ring-white outline-none"
              />
            </div>

            <div>
              <label className="block text-[10px] font-bold text-white/50 uppercase mb-2">Email Address *</label>
              <input
                type="email"
                value={formData.clientEmail}
                onChange={(e) => updateField('clientEmail', e.target.value)}
                placeholder="jane@example.com"
                className="w-full bg-white/10 border-white/20 rounded-lg p-3 text-white placeholder-white/30 focus:ring-2 focus:ring-white outline-none"
              />
            </div>

            <div>
              <label className="block text-[10px] font-bold text-white/50 uppercase mb-2">Phone Number *</label>
              <input
                type="tel"
                value={formData.clientPhone}
                onChange={(e) => updateField('clientPhone', e.target.value)}
                placeholder="+61 400 000 000"
                className="w-full bg-white/10 border-white/20 rounded-lg p-3 text-white placeholder-white/30 focus:ring-2 focus:ring-white outline-none"
              />
            </div>

            <div>
              <label className="block text-[10px] font-bold text-white/50 uppercase mb-2">Preferred Format *</label>
              <div className="flex gap-3">
                <button
                  onClick={() => {
                    updateField('preferInPerson', true);
                    updateField('preferredLocation', '');
                  }}
                  className={`flex-1 p-3 rounded-lg border-2 transition-all ${
                    formData.preferInPerson
                      ? 'bg-white/20 border-white text-white'
                      : 'bg-white/5 border-white/20 text-white/70 hover:border-white/40'
                  }`}
                >
                  In-Person
                </button>
                <button
                  onClick={() => {
                    updateField('preferInPerson', false);
                    updateField('preferredLocation', 'Online');
                  }}
                  className={`flex-1 p-3 rounded-lg border-2 transition-all ${
                    !formData.preferInPerson
                      ? 'bg-white/20 border-white text-white'
                      : 'bg-white/5 border-white/20 text-white/70 hover:border-white/40'
                  }`}
                >
                  Online
                </button>
              </div>
            </div>

            {formData.preferInPerson && (
              <div>
                <label className="block text-[10px] font-bold text-white/50 uppercase mb-2">Location / Postcode *</label>
                <input
                  type="text"
                  value={formData.preferredLocation}
                  onChange={(e) => updateField('preferredLocation', e.target.value)}
                  placeholder="Melbourne 3000"
                  className="w-full bg-white/10 border-white/20 rounded-lg p-3 text-white placeholder-white/30 focus:ring-2 focus:ring-white outline-none"
                />
              </div>
            )}
          </div>

          <button
            onClick={nextStep}
            disabled={!canProceed(1)}
            className="w-full bg-white text-indigo-600 font-bold py-4 rounded-xl shadow-lg transition-all disabled:opacity-50 disabled:cursor-not-allowed hover:scale-105"
          >
            Next: Dispute Details →
          </button>
        </div>
      )}

      {/* Step 2: Dispute Details */}
      {currentStep === 2 && (
        <div className="space-y-6">
          <div>
            <h2 className="text-2xl font-bold mb-2">Dispute Details</h2>
            <p className="text-white/70 text-sm">Help us understand your situation</p>
          </div>

          <div className="space-y-4">
            <div>
              <label className="block text-[10px] font-bold text-white/50 uppercase mb-2">Type of Dispute *</label>
              <select
                value={formData.disputeType}
                onChange={(e) => updateField('disputeType', e.target.value as DisputeType)}
                className="w-full bg-white/10 border-white/20 rounded-lg p-3 text-white focus:ring-2 focus:ring-white outline-none [&>option]:text-gray-900 [&>option]:bg-white"
              >
                <option value="" disabled className="text-gray-400">Select dispute type...</option>
                <option value={DisputeType.FAMILY_PARENTING}>Family Law - Parenting</option>
                <option value={DisputeType.FAMILY_PROPERTY}>Family Law - Property</option>
                <option value={DisputeType.COMMERCIAL}>Commercial Dispute</option>
                <option value={DisputeType.WORKPLACE}>Workplace Conflict</option>
                <option value={DisputeType.COMMUNITY}>Community/Neighborhood</option>
                <option value={DisputeType.OTHER}>Other</option>
              </select>
            </div>

            <div>
              <label className="block text-[10px] font-bold text-white/50 uppercase mb-2">How long has this been ongoing? *</label>
              <input
                type="text"
                value={formData.disputeDuration}
                onChange={(e) => updateField('disputeDuration', e.target.value)}
                placeholder="e.g., 6 months, 2 years"
                className="w-full bg-white/10 border-white/20 rounded-lg p-3 text-white placeholder-white/30 focus:ring-2 focus:ring-white outline-none"
              />
            </div>

            <div>
              <label className="block text-[10px] font-bold text-white/50 uppercase mb-2">Other Party's Name *</label>
              <input
                type="text"
                value={formData.otherPartyName}
                onChange={(e) => updateField('otherPartyName', e.target.value)}
                placeholder="For the Agreement to Mediate"
                className="w-full bg-white/10 border-white/20 rounded-lg p-3 text-white placeholder-white/30 focus:ring-2 focus:ring-white outline-none"
              />
            </div>

            <div>
              <label className="block text-[10px] font-bold text-white/50 uppercase mb-2">Brief Description *</label>
              <textarea
                value={formData.disputeDescription}
                onChange={(e) => updateField('disputeDescription', e.target.value)}
                placeholder="Please provide a brief overview of the situation..."
                rows={4}
                className="w-full bg-white/10 border-white/20 rounded-lg p-3 text-white placeholder-white/30 focus:ring-2 focus:ring-white outline-none resize-none"
              />
              <p className="text-xs text-white/50 mt-1">This will be shared with the mediator</p>
            </div>
          </div>

          <div className="flex gap-3">
            <button
              onClick={prevStep}
              className="flex-1 bg-white/10 text-white font-bold py-4 rounded-xl border-2 border-white/20 hover:bg-white/20 transition-all"
            >
              ← Back
            </button>
            <button
              onClick={nextStep}
              disabled={!canProceed(2)}
              className="flex-1 bg-white text-indigo-600 font-bold py-4 rounded-xl shadow-lg transition-all disabled:opacity-50 disabled:cursor-not-allowed hover:scale-105"
            >
              Next: Legal Context →
            </button>
          </div>
        </div>
      )}

      {/* Step 3: Legal & Safety Context */}
      {currentStep === 3 && (
        <div className="space-y-6">
          <div>
            <h2 className="text-2xl font-bold mb-2">Legal & Safety Context</h2>
            <p className="text-white/70 text-sm">This helps us ensure appropriate support</p>
          </div>

          <div className="space-y-4">
            <div className="space-y-3">
              <label className="flex items-center gap-3 p-3 bg-white/5 rounded-lg cursor-pointer hover:bg-white/10 transition-all">
                <input
                  type="checkbox"
                  checked={formData.hasCourtOrders}
                  onChange={(e) => updateField('hasCourtOrders', e.target.checked)}
                  className="w-5 h-5 rounded accent-white"
                />
                <span className="text-white">Any active court orders?</span>
              </label>

              <label className="flex items-center gap-3 p-3 bg-white/5 rounded-lg cursor-pointer hover:bg-white/10 transition-all">
                <input
                  type="checkbox"
                  checked={formData.hasSafetyOrders}
                  onChange={(e) => updateField('hasSafetyOrders', e.target.checked)}
                  className="w-5 h-5 rounded accent-white"
                />
                <span className="text-white">Any safety or intervention orders?</span>
              </label>

              <label className="flex items-center gap-3 p-3 bg-white/5 rounded-lg cursor-pointer hover:bg-white/10 transition-all">
                <input
                  type="checkbox"
                  checked={formData.hasContracts}
                  onChange={(e) => updateField('hasContracts', e.target.checked)}
                  className="w-5 h-5 rounded accent-white"
                />
                <span className="text-white">Any relevant contracts or agreements?</span>
              </label>
            </div>

            <div>
              <label className="block text-[10px] font-bold text-white/50 uppercase mb-2">Additional Context (optional)</label>
              <textarea
                value={formData.contextDetails}
                onChange={(e) => updateField('contextDetails', e.target.value)}
                placeholder="Any additional legal context we should know about..."
                rows={3}
                className="w-full bg-white/10 border-white/20 rounded-lg p-3 text-white placeholder-white/30 focus:ring-2 focus:ring-white outline-none resize-none"
              />
            </div>

            <div className="border-t border-white/20 pt-4 mt-6">
              <h3 className="font-bold mb-3">Special Considerations</h3>
              
              <label className="flex items-center gap-3 p-3 bg-white/5 rounded-lg cursor-pointer hover:bg-white/10 transition-all mb-3">
                <input
                  type="checkbox"
                  checked={formData.needsDisabilityAccommodation}
                  onChange={(e) => updateField('needsDisabilityAccommodation', e.target.checked)}
                  className="w-5 h-5 rounded accent-white"
                />
                <span className="text-white">I need disability accommodations</span>
              </label>

              <div className="space-y-2">
                <label className="block text-[10px] font-bold text-white/50 uppercase">Cultural or Language Needs</label>
                <input
                  type="text"
                  value={formData.culturalOrLanguageNeeds}
                  onChange={(e) => updateField('culturalOrLanguageNeeds', e.target.value)}
                  placeholder="e.g., interpreter needed, cultural considerations"
                  className="w-full bg-white/10 border-white/20 rounded-lg p-3 text-white placeholder-white/30 focus:ring-2 focus:ring-white outline-none"
                />
              </div>

              <div className="space-y-2 mt-3">
                <label className="block text-[10px] font-bold text-white/50 uppercase">Safety Concerns</label>
                <textarea
                  value={formData.safetyConcerns}
                  onChange={(e) => updateField('safetyConcerns', e.target.value)}
                  placeholder="Any safety concerns we should be aware of..."
                  rows={2}
                  className="w-full bg-white/10 border-white/20 rounded-lg p-3 text-white placeholder-white/30 focus:ring-2 focus:ring-white outline-none resize-none"
                />
              </div>
            </div>
          </div>

          <div className="flex gap-3">
            <button
              onClick={prevStep}
              className="flex-1 bg-white/10 text-white font-bold py-4 rounded-xl border-2 border-white/20 hover:bg-white/20 transition-all"
            >
              ← Back
            </button>
            <button
              onClick={nextStep}
              className="flex-1 bg-white text-indigo-600 font-bold py-4 rounded-xl shadow-lg hover:scale-105 transition-all"
            >
              Review & Proceed →
            </button>
          </div>
        </div>
      )}

      {/* Step 4: Review & Payment */}
      {currentStep === 4 && (
        <div className="space-y-6">
          <div>
            <h2 className="text-2xl font-bold mb-2">Review & Proceed</h2>
            <p className="text-white/70 text-sm">Please review your information</p>
          </div>

          <div className="space-y-4 bg-white/10 rounded-xl p-6 border border-white/20">
            <div>
              <div className="text-xs text-white/50 uppercase font-bold mb-1">Contact</div>
              <div className="text-white">
                <div>{formData.clientName}</div>
                <div className="text-sm text-white/70">{formData.clientEmail}</div>
                <div className="text-sm text-white/70">{formData.clientPhone}</div>
                <div className="text-sm text-white/70">{formData.preferInPerson ? formData.preferredLocation : 'Online'}</div>
              </div>
            </div>

            <div className="border-t border-white/10 pt-4">
              <div className="text-xs text-white/50 uppercase font-bold mb-1">Dispute</div>
              <div className="text-white">
                <div className="text-sm">{formData.disputeType.replace(/_/g, ' ')}</div>
                <div className="text-sm text-white/70">Duration: {formData.disputeDuration}</div>
                <div className="text-sm text-white/70">Other party: {formData.otherPartyName}</div>
              </div>
            </div>
          </div>

          <div className="bg-green-900/30 border border-green-500/30 rounded-xl p-6">
            <h3 className="font-bold text-lg mb-3 text-white">What happens next?</h3>
            <ul className="space-y-2 text-sm text-white/90">
              <li className="flex items-start gap-2">
                <span className="text-green-400 mt-0.5">✓</span>
                <span>{mediator.displayName} will receive your request and contact you within 48 hours</span>
              </li>
              <li className="flex items-start gap-2">
                <span className="text-green-400 mt-0.5">✓</span>
                <span>They'll send you the Agreement to Mediate to review</span>
              </li>
              <li className="flex items-start gap-2">
                <span className="text-green-400 mt-0.5">✓</span>
                <span>You'll schedule your pre-mediation session together</span>
              </li>
            </ul>
          </div>

          <div className="bg-white/10 rounded-xl p-6 border border-white/20">
            <div className="flex justify-between items-center mb-2">
              <span className="text-white/70">Pre-Mediation Fee</span>
              <span className="text-2xl font-bold text-white">{formatAud(estimatedFeeCents)}</span>
            </div>
            <p className="text-xs text-white/60">This includes a 1-hour pre-mediation consultation</p>
          </div>

          <div className="flex gap-3">
            <button
              onClick={prevStep}
              className="flex-1 bg-white/10 text-white font-bold py-4 rounded-xl border-2 border-white/20 hover:bg-white/20 transition-all"
            >
              ← Back
            </button>
            <button
              onClick={handleProceedToPayment}
              disabled={submitting}
              className="flex-1 bg-green-600 hover:bg-green-700 text-white font-bold py-4 rounded-xl shadow-lg transition-all disabled:opacity-50"
            >
              {submitting ? 'Processing...' : `Pay ${formatAud(estimatedFeeCents)}`}
            </button>
          </div>
        </div>
      )}
    </div>
  );
};

export default EngagementFormComponent;
