
import React, { useState, useEffect } from 'react';
import { Link } from 'react-router-dom';
import { MediatorProfilePublic, MediationCategory, PriceBand } from '../../types';
import { SEEDED_CATEGORIES } from '../../services/SeedService';
import { MediatorService } from '../../services/MediatorService';
import { AvailabilityService } from '../../services/AvailabilityService';
import { suggestMediators, SuggestedMediator } from '../../lib/suggest';
import { formatAud, dollarsToCents } from '../../lib/money';
import { getPricingSummary } from '../../lib/pricing';

const MediatorDirectory: React.FC = () => {
  const [mediators, setMediators] = useState<MediatorProfilePublic[]>([]);
  const [unavailableMediatorIds, setUnavailableMediatorIds] = useState<string[]>([]);
  const [selectedCategory, setSelectedCategory] = useState<string>('all');
  const [expectedHours, setExpectedHours] = useState<number>(3);
  const [partyCount, setPartyCount] = useState<number>(2);
  const [onlineOnly, setOnlineOnly] = useState<boolean>(false);
  const [fixedFeeOnly, setFixedFeeOnly] = useState<boolean>(false);
  const [priceBand, setPriceBand] = useState<PriceBand | ''>('');
  const [maxBudget, setMaxBudget] = useState<string>('');
  const [loading, setLoading] = useState(true);

  useEffect(() => {
    loadMediators();
  }, []);

  const loadMediators = async () => {
    setLoading(true);
    const data = await MediatorService.getPublishedMediators();
    const unavailable = await AvailabilityService.getUnavailableMediators();
    setMediators(data);
    setUnavailableMediatorIds(unavailable);
    setLoading(false);
  };

  const handleMaxBudgetChange = (val: string) => {
    const num = Number(val);
    if (isNaN(num)) return;
    const clamped = num < 0 ? "0" : val;
    setMaxBudget(clamped);
  };

  const resetFilters = () => {
    setSelectedCategory('all');
    setExpectedHours(3);
    setPartyCount(2);
    setOnlineOnly(false);
    setFixedFeeOnly(false);
    setPriceBand('');
    setMaxBudget('');
  };

  let filteredMediators = mediators;
  
  // Filter out unavailable mediators
  filteredMediators = filteredMediators.filter(
    m => !unavailableMediatorIds.includes(m.id)
  );
  
  if (fixedFeeOnly) {
    filteredMediators = filteredMediators.filter(m => m.optInFixedFee);
  }

  const suggestions = suggestMediators(filteredMediators, {
    categoryId: selectedCategory,
    expectedHours,
    partyCount,
    onlineOnly,
    priceBand: priceBand || undefined,
    maxBudgetCents: maxBudget ? dollarsToCents(Number(maxBudget)) : undefined
  });

  const topSuggestions = suggestions.slice(0, 3);
  const otherResults = suggestions.slice(3);

  return (
    <div className="max-w-7xl mx-auto px-4 py-12">
      {/* Header with Help Button */}
      <div className="flex items-center justify-between mb-8">
        <div>
          <h1 className="text-3xl font-bold mb-2">Find a Mediator</h1>
          <p className="text-slate-600">Browse our directory of qualified mediation professionals</p>
        </div>
        <Link 
          to="/request-callback"
          className="bg-indigo-100 hover:bg-indigo-200 text-indigo-700 px-6 py-3 rounded-lg font-bold transition-all flex items-center gap-2 shadow-sm"
        >
          <svg className="w-5 h-5" fill="none" stroke="currentColor" viewBox="0 0 24 24">
            <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M8.228 9c.549-1.165 2.03-2 3.772-2 2.21 0 4 1.343 4 3 0 1.4-1.278 2.575-3.006 2.907-.542.104-.994.54-.994 1.093m0 3h.01M21 12a9 9 0 11-18 0 9 9 0 0118 0z" />
          </svg>
          Need Help Choosing?
        </Link>
      </div>

      <div className="flex flex-col lg:flex-row gap-8">
        {/* Sidebar Filters */}
        <div className="lg:w-1/4">
          <div className="bg-white p-6 rounded-xl shadow-sm border border-slate-200 sticky top-24">
            <div className="flex items-center justify-between mb-6">
              <h2 className="text-lg font-bold">Search Filters</h2>
              <button 
                onClick={resetFilters}
                className="text-xs text-indigo-600 hover:opacity-80 font-bold uppercase tracking-wider"
              >
                Reset
              </button>
            </div>
            
            <div className="space-y-5">
              <div>
                <label className="block text-sm font-semibold text-slate-700 mb-2">Category</label>
                <select 
                  value={selectedCategory} 
                  onChange={(e) => setSelectedCategory(e.target.value)}
                  className="w-full border rounded-lg p-2 focus:ring-2 focus:ring-indigo-500 text-sm outline-none"
                >
                  <option value="all">All Categories</option>
                  {SEEDED_CATEGORIES.map(c => <option key={c.id} value={c.id}>{c.name}</option>)}
                </select>
              </div>

              <div className="grid grid-cols-2 gap-3">
                <div>
                  <label className="block text-sm font-semibold text-slate-700 mb-2">Hours</label>
                  <input 
                    type="number" min="1"
                    value={expectedHours} 
                    onChange={(e) => setExpectedHours(Math.max(1, Number(e.target.value)))}
                    className="w-full border rounded-lg p-2 text-sm outline-none focus:ring-2 focus:ring-indigo-500"
                  />
                </div>
                <div>
                  <label className="block text-sm font-semibold text-slate-700 mb-2">Parties</label>
                  <input 
                    type="number" min="2"
                    value={partyCount} 
                    onChange={(e) => setPartyCount(Math.max(2, Number(e.target.value)))}
                    className="w-full border rounded-lg p-2 text-sm outline-none focus:ring-2 focus:ring-indigo-500"
                  />
                </div>
              </div>

              <div>
                <label className="block text-sm font-semibold text-slate-700 mb-2">Max Budget ($ AUD)</label>
                <input 
                  type="number" 
                  value={maxBudget} 
                  onChange={(e) => handleMaxBudgetChange(e.target.value)}
                  placeholder="e.g. 2000"
                  className="w-full border rounded-lg p-2 text-sm outline-none focus:ring-2 focus:ring-indigo-500"
                />
              </div>

              <div className="space-y-3 pt-2">
                <label className="flex items-center gap-3 cursor-pointer group">
                  <input 
                    type="checkbox" 
                    checked={onlineOnly} 
                    onChange={(e) => setOnlineOnly(e.target.checked)}
                    className="w-4 h-4 text-indigo-600 border-slate-300 rounded focus:ring-indigo-500"
                  />
                  <span className="text-sm font-medium text-slate-700 group-hover:text-indigo-600 transition-colors">Online available</span>
                </label>

                <label className="flex items-center gap-3 cursor-pointer group">
                  <input 
                    type="checkbox" 
                    checked={fixedFeeOnly} 
                    onChange={(e) => setFixedFeeOnly(e.target.checked)}
                    className="w-4 h-4 text-indigo-600 border-slate-300 rounded focus:ring-indigo-500"
                  />
                  <span className="text-sm font-medium text-slate-700 group-hover:text-indigo-600 transition-colors">Verified Fixed Fee</span>
                </label>
              </div>

              <button 
                onClick={resetFilters}
                className="w-full mt-4 py-2 border border-slate-200 rounded-lg text-xs font-bold text-slate-500 hover:bg-slate-50 transition-colors uppercase tracking-widest"
              >
                Clear All Filters
              </button>
            </div>
          </div>
        </div>

        {/* Content Area */}
        <div className="lg:w-3/4 space-y-12">
          {loading ? (
            <div className="py-20 text-center text-slate-400">Loading mediators...</div>
          ) : (
            <>
              {topSuggestions.length > 0 ? (
                <section>
                  <div className="flex items-center justify-between mb-6">
                    <h2 className="text-2xl font-bold">Suggested Mediators</h2>
                    <span className="text-slate-400 text-sm italic">Top 3 matches by value</span>
                  </div>
                  <div className="grid md:grid-cols-3 gap-6">
                    {topSuggestions.map((item) => (
                      <MediatorCard key={item.profile.id} item={item} highlight />
                    ))}
                  </div>
                </section>
              ) : (
                <div className="bg-slate-50 border-2 border-dashed border-slate-200 rounded-2xl p-12 text-center text-slate-400">
                  No mediators found matching these criteria. Try adjusting your filters.
                </div>
              )}

              {otherResults.length > 0 && (
                <section>
                  <h2 className="text-2xl font-bold mb-6">Other Matches</h2>
                  <div className="grid md:grid-cols-2 gap-6">
                    {otherResults.map((item) => (
                      <MediatorCard key={item.profile.id} item={item} />
                    ))}
                  </div>
                </section>
              )}
            </>
          )}

          <div className="bg-amber-50 border border-amber-200 p-4 rounded-xl flex gap-3 text-amber-800">
            <svg xmlns="http://www.w3.org/2000/svg" className="h-5 w-5 flex-shrink-0" viewBox="0 0 20 20" fill="currentColor">
              <path fillRule="evenodd" d="M8.257 3.099c.765-1.36 2.722-1.36 3.486 0l5.58 9.92c.75 1.334-.213 2.98-1.742 2.98H4.42c-1.53 0-2.493-1.646-1.743-2.98l5.58-9.92zM11 13a1 1 0 11-2 0 1 1 0 012 0zm-1-8a1 1 0 00-1-1v3a1 1 0 002 0V6a1 1 0 00-1-1z" clipRule="evenodd" />
            </svg>
            <p className="text-sm">
              <strong>Indicative estimate only:</strong> Final fees confirmed after assessment. Prices include GST.
            </p>
          </div>
        </div>
      </div>
    </div>
  );
};

const MediatorCard: React.FC<{ item: SuggestedMediator; highlight?: boolean }> = ({ item, highlight }) => {
  const { profile, estimatedTotalCents, priceBand } = item;
  
  return (
    <div className={`bg-white rounded-xl shadow-sm border group transition-all duration-300 hover:shadow-md ${highlight ? 'border-indigo-200 ring-4 ring-indigo-50' : 'border-slate-200'}`}>
      <div className="p-5">
        <div className="flex gap-4 mb-4">
          <img src={profile.photoUrl} alt={profile.displayName} className="w-16 h-16 rounded-full object-cover border-2 border-slate-50" />
          <div className="flex-grow min-w-0">
            <div className="flex items-center gap-2">
              <h3 className="font-bold text-lg group-hover:text-indigo-600 transition-colors truncate">{profile.displayName}</h3>
              {profile.optInFixedFee && (
                <span title="Verified Fixed Fee Program" className="text-indigo-600">
                  <svg xmlns="http://www.w3.org/2000/svg" className="h-5 w-5" viewBox="0 0 20 20" fill="currentColor">
                    <path fillRule="evenodd" d="M6.267 3.455a3.066 3.066 0 001.745-.723 3.066 3.066 0 013.976 0 3.066 3.066 0 001.745.723 3.066 3.066 0 012.812 2.812c.051.643.304 1.254.723 1.745a3.066 3.066 0 010 3.976 3.066 3.066 0 00-.723 1.745 3.066 3.066 0 01-2.812 2.812 3.066 3.066 0 00-1.745.723 3.066 3.066 0 01-3.976 0 3.066 3.066 0 00-1.745-.723 3.066 3.066 0 01-2.812-2.812 3.066 3.066 0 00-.723-1.745 3.066 3.066 0 010-3.976 3.066 3.066 0 00.723-1.745 3.066 3.066 0 012.812-2.812zm7.44 5.252a1 1 0 00-1.414-1.414L9 10.586 7.707 9.293a1 1 0 00-1.414 1.414l2 2a1 1 0 001.414 0l4-4z" clipRule="evenodd" />
                  </svg>
                </span>
              )}
            </div>
            <p className="text-xs text-slate-500 line-clamp-1">{profile.headline}</p>
            <p className="text-[10px] text-slate-400 truncate mt-0.5 font-medium italic">{getPricingSummary(profile.pricing)}</p>
          </div>
        </div>
        
        <div className="space-y-3 mb-6">
          <p className="text-xs text-slate-600 line-clamp-2 leading-relaxed">{profile.bio}</p>
        </div>

        <div className="pt-4 border-t border-slate-50 flex items-end justify-between">
          <div className="min-w-0">
            <div className="text-[10px] text-slate-400 uppercase font-bold tracking-tight">Indicative Estimate</div>
            <div className="text-xl font-bold text-indigo-900">{formatAud(estimatedTotalCents)}</div>
          </div>
          <Link 
            to={`/mediators/${profile.id}`} 
            className="bg-indigo-600 text-white px-4 py-2 rounded-lg text-sm font-bold hover:bg-indigo-700 transition-colors shadow-sm whitespace-nowrap"
          >
            View Profile
          </Link>
        </div>
      </div>
    </div>
  );
};

export default MediatorDirectory;
