
import React, { useState, useEffect } from 'react';
import { useParams, useNavigate } from 'react-router-dom';
import { 
  MediatorProfilePublic, 
  BookingStatus, 
  BookingRequest, 
  PaymentKind, 
  FixedFeePackage,
  ESignStatus
} from '../../types';
import { MediatorService } from '../../services/MediatorService';
import { BookingService } from '../../services/BookingService';
import { FixedFeeService } from '../../services/FixedFeeService';
import { ESignService } from '../../services/ESignService';
import { AvailabilityService } from '../../services/AvailabilityService';
import { SEEDED_CATEGORIES } from '../../services/SeedService';
import { formatAud } from '../../lib/money';
import { estimateTotalCents, getPriceBand } from '../../lib/pricing';
import EngagementFormComponent from './EngagementForm';

const MediatorProfileView: React.FC = () => {
  const { id } = useParams<{ id: string }>();
  const navigate = useNavigate();
  const [profile, setProfile] = useState<MediatorProfilePublic | null>(null);
  const [loading, setLoading] = useState(true);
  const [isAvailable, setIsAvailable] = useState(true);
  const [returnDate, setReturnDate] = useState<number | null>(null);
  
  // Package handling
  const [selectedPkg, setSelectedPkg] = useState<FixedFeePackage | null>(null);

  // Estimation state
  const [estHours, setEstHours] = useState(3);
  const [estParties, setEstParties] = useState(2);

  // Booking state
  const [showEngagementForm, setShowEngagementForm] = useState(false);

  useEffect(() => {
    if (id) {
      MediatorService.getMediatorById(id).then(res => {
        setProfile(res);
        setLoading(false);
      });
      
      // Check availability
      AvailabilityService.getMediatorAvailability(id).then(avail => {
        setIsAvailable(avail.status === 'AVAILABLE');
        if (avail.currentPeriod) {
          setReturnDate(avail.currentPeriod.endDate);
        }
      });
    }
  }, [id]);

  if (loading) return <div className="p-20 text-center text-slate-500">Loading profile...</div>;
  if (!profile) return <div className="p-20 text-center text-red-500">Mediator not found.</div>;

  // Use promotional rate if available, otherwise use standard rate
  const actualPreMediationFee = profile.pricing?.hasPromotionalOffer && profile.pricing?.promotionalPreMediationFeeCents
    ? profile.pricing.promotionalPreMediationFeeCents
    : profile.pricing.preMediationFeeCents;

  const currentEstimate = selectedPkg 
    ? selectedPkg.totalFeeCents 
    : estimateTotalCents({
        pricing: profile.pricing,
        expectedHours: estHours,
        partyCount: estParties,
      });

  return (
    <div className="max-w-7xl mx-auto px-4 py-12">
      <div className="flex flex-col lg:flex-row gap-8">
        {/* Profile Content */}
        <div className="lg:w-2/3 space-y-8">
          <section className="bg-white p-8 rounded-2xl shadow-sm border border-slate-100 flex flex-col md:flex-row gap-8 items-start">
            <img src={profile.photoUrl} className="w-48 h-48 rounded-2xl object-cover border-4 border-slate-50 shadow-sm" />
            <div className="flex-grow">
              <div className="flex items-center gap-3 mb-2">
                <h1 className="text-3xl font-bold">{profile.displayName}</h1>
                {profile.optInFixedFee && (
                  <span className="bg-indigo-600 text-white text-[10px] font-extrabold px-2 py-1 rounded uppercase flex items-center gap-1">
                    <svg className="h-3 w-3" fill="currentColor" viewBox="0 0 20 20"><path fillRule="evenodd" d="M6.267 3.455a3.066 3.066 0 001.745-.723 3.066 3.066 0 013.976 0 3.066 3.066 0 001.745.723 3.066 3.066 0 012.812 2.812c.051.643.304 1.254.723 1.745a3.066 3.066 0 010 3.976 3.066 3.066 0 00-.723 1.745 3.066 3.066 0 01-2.812 2.812 3.066 3.066 0 00-1.745.723 3.066 3.066 0 01-3.976 0 3.066 3.066 0 00-1.745-.723 3.066 3.066 0 01-2.812-2.812 3.066 3.066 0 00-.723-1.745 3.066 3.066 0 010-3.976 3.066 3.066 0 00.723-1.745 3.066 3.066 0 012.812-2.812zm7.44 5.252a1 1 0 00-1.414-1.414L9 10.586 7.707 9.293a1 1 0 00-1.414 1.414l2 2a1 1 0 001.414 0l4-4z" clipRule="evenodd" /></svg>
                    Verified Fixed Fee
                  </span>
                )}
              </div>
              <p className="text-xl text-indigo-600 font-medium mb-4">{profile.headline}</p>
              <div className="flex flex-wrap gap-2 mb-6">
                {profile.categories.map(catId => (
                  <span key={catId} className="bg-indigo-50 text-indigo-700 px-3 py-1 rounded-full text-xs font-semibold border border-indigo-100">
                    {SEEDED_CATEGORIES.find(c => c.id === catId)?.name || catId}
                  </span>
                ))}
              </div>
              <p className="text-slate-600 leading-relaxed whitespace-pre-wrap">{profile.bio}</p>
            </div>
          </section>

          {profile.optInFixedFee && profile.activePackages.length > 0 && (
            <section className="bg-white p-8 rounded-2xl shadow-sm border border-indigo-100">
              <h2 className="text-xl font-bold mb-6 text-indigo-900">Standardized Online Packages</h2>
              <div className="grid gap-4">
                {profile.activePackages.map(pkgId => {
                  const pkg = FixedFeeService.getPackageById(pkgId);
                  if (!pkg) return null;
                  const isSelected = selectedPkg?.id === pkg.id;
                  return (
                    <div 
                      key={pkg.id}
                      onClick={() => setSelectedPkg(isSelected ? null : pkg)}
                      className={`p-5 rounded-xl border-2 transition-all cursor-pointer flex justify-between items-center ${
                        isSelected 
                        ? 'bg-indigo-50 border-indigo-600 ring-4 ring-indigo-50 shadow-md' 
                        : 'border-slate-100 hover:border-indigo-200'
                      }`}
                    >
                      <div>
                        <div className="font-bold text-lg">{pkg.name}</div>
                        <div className="text-sm text-slate-500">{pkg.description}</div>
                        <div className="mt-2 text-[10px] font-bold text-indigo-600 uppercase tracking-widest">{pkg.durationHours} Hours Session</div>
                      </div>
                      <div className="text-right">
                        <div className="text-2xl font-black text-indigo-900">{formatAud(pkg.totalFeeCents)}</div>
                        <div className="text-[10px] text-slate-400 font-bold">TOTAL INC. GST</div>
                      </div>
                    </div>
                  );
                })}
              </div>
            </section>
          )}

          <section className="bg-white p-8 rounded-2xl shadow-sm border border-slate-100">
            <h2 className="text-xl font-bold mb-6">Expertise & Background</h2>
            <div className="grid md:grid-cols-2 gap-8">
              <div>
                <h3 className="text-sm uppercase tracking-wider text-slate-400 font-bold mb-3">Qualifications</h3>
                <ul className="space-y-2">
                  {profile.qualifications.map((q, i) => (
                    <li key={i} className="flex gap-2 items-center text-slate-700">
                      <svg className="h-4 w-4 text-green-500" fill="none" viewBox="0 0 24 24" stroke="currentColor">
                        <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={3} d="M5 13l4 4L19 7" />
                      </svg>
                      {q}
                    </li>
                  ))}
                </ul>
              </div>
              <div>
                <h3 className="text-sm uppercase tracking-wider text-slate-400 font-bold mb-3">Service Areas</h3>
                <div className="flex flex-wrap gap-2">
                  {profile.locations.map((l, i) => (
                    <span key={i} className="bg-slate-100 px-3 py-1 rounded-md text-slate-600 text-sm">{l}</span>
                  ))}
                  {profile.onlineAvailable && <span className="bg-green-50 text-green-700 px-3 py-1 rounded-md text-sm border border-green-100 font-semibold">Online/Video Available</span>}
                </div>
              </div>
            </div>
          </section>
        </div>

        {/* Action Sidebar */}
        <div className="lg:w-1/3">
          <div className="bg-indigo-600 text-white rounded-2xl p-8 shadow-xl sticky top-24">
            {!showEngagementForm && (
              <>
                <h2 className="text-2xl font-bold mb-6">
                  {selectedPkg ? 'Fixed Fee Quote' : 'Indicative Estimator'}
                </h2>
                
                {!selectedPkg ? (
                  <div className="space-y-5 mb-8">
                    <div>
                      <div className="flex justify-between items-center mb-2">
                        <label className="block text-xs font-bold text-white/50 uppercase">Duration (Hours)</label>
                        <span className="text-sm font-bold text-white">{estHours} Hours</span>
                      </div>
                      <input 
                        type="range" min="1" max="16" value={estHours} 
                        onChange={(e) => setEstHours(Math.max(1, Number(e.target.value)))}
                        className="w-full accent-white"
                      />
                    </div>
                    <div>
                      <label className="block text-xs font-bold text-white/50 uppercase mb-2">No. of Parties</label>
                      <select 
                        value={estParties} onChange={(e) => setEstParties(Math.max(2, Number(e.target.value)))}
                        className="w-full bg-white/10 border-white/20 rounded p-2 text-sm focus:ring-2 focus:ring-white outline-none text-white"
                      >
                        {[2,3,4,5,6,7,8].map(n => <option key={n} value={n} className="text-slate-900">{n} Parties</option>)}
                      </select>
                    </div>
                  </div>
                ) : (
                  <div className="mb-8 p-4 bg-white/10 rounded-xl border border-white/20">
                    <div className="text-[10px] font-bold text-white/50 uppercase">Package Selected</div>
                    <div className="font-bold text-lg">{selectedPkg.name}</div>
                    <div className="text-xs text-indigo-100/70 mt-1">Standardized Online Mediation</div>
                  </div>
                )}

                <div className="pt-6 border-t border-white/10 mb-8">
                  <div className="flex items-end justify-between">
                    <div>
                      <div className="text-xs font-bold text-white/50 uppercase">{selectedPkg ? 'Guaranteed Fee' : 'Total Estimate'}</div>
                      <div className="text-3xl font-bold text-white">{formatAud(currentEstimate)}</div>
                    </div>
                    {!selectedPkg && <span className={`px-2 py-0.5 rounded text-[10px] font-bold border border-white/20`}>{getPriceBand(currentEstimate)}</span>}
                  </div>
                </div>

                {/* Promotional Offer Badge */}
                {profile.pricing?.hasPromotionalOffer && profile.pricing?.promotionalPreMediationFeeCents && !selectedPkg && (
                  <div className="bg-green-500/20 border border-green-500/30 rounded-lg p-4 mb-4">
                    <div className="flex items-start gap-3 text-green-100">
                      <svg className="w-5 h-5 mt-0.5 flex-shrink-0" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                        <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M12 8c-1.657 0-3 .895-3 2s1.343 2 3 2 3 .895 3 2-1.343 2-3 2m0-8c1.11 0 2.08.402 2.599 1M12 8V7m0 1v8m0 0v1m0-1c-1.11 0-2.08-.402-2.599-1M21 12a9 9 0 11-18 0 9 9 0 0118 0z" />
                      </svg>
                      <div className="flex-1">
                        <p className="font-semibold text-white">💰 Special Offer Available</p>
                        <p className="text-sm text-white/90 mt-1">
                          {profile.pricing.promotionalOfferDescription || 'Initial consultation at promotional rate'}
                        </p>
                        <div className="mt-2 flex items-baseline gap-2">
                          <span className="text-2xl font-bold text-white">{formatAud(profile.pricing.promotionalPreMediationFeeCents)}</span>
                          <span className="text-sm text-white/70 line-through">{formatAud(profile.pricing.preMediationFeeCents)}</span>
                          <span className="text-xs bg-green-600 text-white px-2 py-1 rounded font-bold">
                            SAVE {Math.round((1 - profile.pricing.promotionalPreMediationFeeCents / profile.pricing.preMediationFeeCents) * 100)}%
                          </span>
                        </div>
                        <p className="text-xs text-white/70 mt-2">
                          This offer is available to all parties in your mediation
                        </p>
                      </div>
                    </div>
                  </div>
                )}

                {!isAvailable && returnDate && (
                  <div className="bg-yellow-500/20 border border-yellow-500/30 rounded-lg p-4 mb-4">
                    <div className="flex items-center gap-2 text-yellow-100">
                      <svg className="w-5 h-5" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                        <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M12 8v4m0 4h.01M21 12a9 9 0 11-18 0 9 9 0 0118 0z" />
                      </svg>
                      <div>
                        <p className="font-semibold">Currently Unavailable</p>
                        <p className="text-sm text-white/80">
                          Returns {new Date(returnDate).toLocaleDateString('en-AU', { day: 'numeric', month: 'short', year: 'numeric' })}
                        </p>
                      </div>
                    </div>
                  </div>
                )}

                <button 
                  onClick={() => setShowEngagementForm(true)}
                  disabled={!isAvailable}
                  className={`w-full font-bold py-4 rounded-xl shadow-lg transition-all active:scale-95 flex items-center justify-center gap-2 ${
                    isAvailable 
                      ? 'bg-white text-indigo-600 hover:bg-gray-50' 
                      : 'bg-gray-400 text-gray-600 cursor-not-allowed opacity-60'
                  }`}
                >
                  {isAvailable ? 'Choose this Mediator' : 'Currently Unavailable'}
                  {isAvailable && (
                    <svg xmlns="http://www.w3.org/2000/svg" className="h-5 w-5" viewBox="0 0 20 20" fill="currentColor">
                      <path fillRule="evenodd" d="M10.293 3.293a1 1 0 011.414 0l6 6a1 1 0 010 1.414l-6 6a1 1 0 01-1.414-1.414L14.586 11H3a1 1 0 110-2h11.586l-4.293-4.293a1 1 0 010-1.414z" clipRule="evenodd" />
                    </svg>
                  )}
                </button>
              </>
            )}

            {showEngagementForm && (
              <EngagementFormComponent
                mediator={profile}
                estimatedFeeCents={actualPreMediationFee}
                onBack={() => setShowEngagementForm(false)}
              />
            )}
          </div>
        </div>
      </div>
    </div>
  );
};

export default MediatorProfileView;
