import React, { useEffect, useState } from 'react';
import { useNavigate, useParams } from 'react-router-dom';
import { IntakeService } from '../../services/IntakeService';
import { IntakeRequest } from '../../types';

const StaffIntakeDetail: React.FC = () => {
  const { id } = useParams<{ id: string }>();
  const navigate = useNavigate();
  const [request, setRequest] = useState<IntakeRequest | null>(null);
  const [loading, setLoading] = useState(true);
  const [notes, setNotes] = useState('');
  const [saving, setSaving] = useState(false);

  // Mock current user - in production, get from auth
  const currentOfficerId = 'officer-001';

  useEffect(() => {
    loadRequest();
  }, [id]);

  const loadRequest = async () => {
    if (!id) return;
    setLoading(true);
    const req = await IntakeService.getById(id);
    if (req) {
      setRequest(req);
      setNotes(req.officerNotes || '');
    }
    setLoading(false);
  };

  const handleClaim = async () => {
    if (!id) return;
    setSaving(true);
    await IntakeService.assignToOfficer(id, currentOfficerId);
    await loadRequest();
    setSaving(false);
  };

  const handleSaveNotes = async () => {
    if (!id) return;
    setSaving(true);
    await IntakeService.updateNotes(id, notes);
    await loadRequest();
    setSaving(false);
  };

  const handleComplete = async () => {
    if (!id) return;
    if (!confirm('Mark this request as completed?')) return;
    
    setSaving(true);
    await IntakeService.completeRequest(id);
    navigate('/staff/intake');
  };

  const handleCancel = async () => {
    if (!id) return;
    if (!confirm('Cancel this request?')) return;
    
    setSaving(true);
    // In a full implementation, you'd have a cancelRequest method
    // For now, just go back
    navigate('/staff/intake');
  };

  const getTimeSince = (timestamp: number): string => {
    const minutes = Math.floor((Date.now() - timestamp) / 60000);
    if (minutes < 60) return `${minutes} minutes ago`;
    const hours = Math.floor(minutes / 60);
    if (hours < 24) return `${hours} hour${hours > 1 ? 's' : ''} ago`;
    const days = Math.floor(hours / 24);
    return `${days} day${days > 1 ? 's' : ''} ago`;
  };

  const isSLABreached = (timestamp: number): boolean => {
    const hours = (Date.now() - timestamp) / (1000 * 60 * 60);
    return hours > 4;
  };

  if (loading) {
    return (
      <div className="flex items-center justify-center min-h-screen">
        <div className="text-center">
          <div className="inline-block w-8 h-8 border-4 border-indigo-600 border-t-transparent rounded-full animate-spin"></div>
          <p className="mt-4 text-slate-600">Loading request...</p>
        </div>
      </div>
    );
  }

  if (!request) {
    return (
      <div className="flex items-center justify-center min-h-screen">
        <div className="text-center">
          <p className="text-slate-600">Request not found</p>
          <button
            onClick={() => navigate('/staff/intake')}
            className="mt-4 text-indigo-600 hover:text-indigo-700"
          >
            ← Back to Queue
          </button>
        </div>
      </div>
    );
  }

  const isAssignedToMe = request.assignedOfficerId === currentOfficerId;
  const canClaim = request.status === 'PENDING' && !request.assignedOfficerId;

  return (
    <div className="min-h-screen bg-gray-50 py-8">
      <div className="max-w-4xl mx-auto px-4 sm:px-6 lg:px-8">
        {/* Back Button */}
        <button
          onClick={() => navigate('/staff/intake')}
          className="flex items-center text-slate-600 hover:text-slate-900 mb-6"
        >
          <svg className="w-5 h-5 mr-2" fill="none" stroke="currentColor" viewBox="0 0 24 24">
            <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M15 19l-7-7 7-7" />
          </svg>
          Back to Queue
        </button>

        {/* Header */}
        <div className="bg-white rounded-lg shadow-md p-6 mb-6">
          <div className="flex items-start justify-between mb-4">
            <div>
              <h1 className="text-2xl font-bold text-slate-900">{request.clientName}</h1>
              <p className="text-sm text-slate-500 mt-1">
                Requested {getTimeSince(request.createdAt)}
              </p>
            </div>
            <div className="flex gap-2">
              {request.status === 'PENDING' && (
                <span className="px-3 py-1 bg-yellow-100 text-yellow-800 rounded-full text-sm font-medium">
                  Pending
                </span>
              )}
              {request.status === 'IN_PROGRESS' && (
                <span className="px-3 py-1 bg-blue-100 text-blue-800 rounded-full text-sm font-medium">
                  In Progress
                </span>
              )}
              {request.status === 'COMPLETED' && (
                <span className="px-3 py-1 bg-green-100 text-green-800 rounded-full text-sm font-medium">
                  Completed
                </span>
              )}
            </div>
          </div>

          {/* SLA Warning */}
          {request.status === 'PENDING' && isSLABreached(request.createdAt) && (
            <div className="bg-red-50 border-l-4 border-red-500 p-4 mb-4">
              <div className="flex items-center">
                <svg className="w-5 h-5 text-red-500 mr-2" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                  <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M12 8v4m0 4h.01M21 12a9 9 0 11-18 0 9 9 0 0118 0z" />
                </svg>
                <span className="text-red-700 font-medium">
                  ⚠️ SLA Breached - Over 4 hours without response
                </span>
              </div>
            </div>
          )}

          {/* Contact Information */}
          <div className="grid grid-cols-1 md:grid-cols-2 gap-4">
            <div>
              <div className="text-sm font-medium text-slate-600 mb-1">Email</div>
              <a href={`mailto:${request.clientEmail}`} className="text-indigo-600 hover:text-indigo-700">
                {request.clientEmail}
              </a>
            </div>
            <div>
              <div className="text-sm font-medium text-slate-600 mb-1">Phone</div>
              <a href={`tel:${request.clientPhone}`} className="text-indigo-600 hover:text-indigo-700">
                {request.clientPhone}
              </a>
            </div>
          </div>
        </div>

        {/* Description */}
        {request.description && (
          <div className="bg-white rounded-lg shadow-md p-6 mb-6">
            <h2 className="text-lg font-semibold text-slate-900 mb-3">Client's Request</h2>
            <p className="text-slate-700 whitespace-pre-wrap">{request.description}</p>
          </div>
        )}

        {/* Claim Section */}
        {canClaim && (
          <div className="bg-indigo-50 border border-indigo-200 rounded-lg p-6 mb-6">
            <div className="flex items-center justify-between">
              <div>
                <h3 className="text-lg font-semibold text-slate-900 mb-2">Claim This Request</h3>
                <p className="text-slate-600">
                  Click below to assign this request to yourself and start working on it.
                </p>
              </div>
              <button
                onClick={handleClaim}
                disabled={saving}
                className="bg-indigo-600 hover:bg-indigo-700 text-white px-6 py-3 rounded-lg font-semibold disabled:opacity-50 whitespace-nowrap ml-4"
              >
                {saving ? 'Claiming...' : 'Claim Request'}
              </button>
            </div>
          </div>
        )}

        {/* Assignment Info */}
        {request.assignedOfficerId && (
          <div className="bg-white rounded-lg shadow-md p-6 mb-6">
            <h2 className="text-lg font-semibold text-slate-900 mb-3">Assignment</h2>
            <div className="flex items-center text-slate-700">
              <svg className="w-5 h-5 mr-2" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M16 7a4 4 0 11-8 0 4 4 0 018 0zM12 14a7 7 0 00-7 7h14a7 7 0 00-7-7z" />
              </svg>
              {isAssignedToMe ? (
                <span className="font-medium text-green-700">Assigned to you</span>
              ) : (
                <span>Assigned to Officer #{request.assignedOfficerId.slice(-4)}</span>
              )}
            </div>
          </div>
        )}

        {/* Officer Notes */}
        {(isAssignedToMe || request.status !== 'PENDING') && (
          <div className="bg-white rounded-lg shadow-md p-6 mb-6">
            <h2 className="text-lg font-semibold text-slate-900 mb-3">Officer Notes</h2>
            <textarea
              value={notes}
              onChange={(e) => setNotes(e.target.value)}
              disabled={!isAssignedToMe || request.status === 'COMPLETED'}
              className="w-full border border-slate-300 rounded-lg p-3 min-h-[150px] disabled:bg-gray-50 disabled:text-gray-600"
              placeholder="Add notes about your conversation with the client, mediator recommendations, etc..."
            />
            {isAssignedToMe && request.status !== 'COMPLETED' && (
              <div className="mt-3 flex justify-end">
                <button
                  onClick={handleSaveNotes}
                  disabled={saving}
                  className="bg-indigo-600 hover:bg-indigo-700 text-white px-4 py-2 rounded-lg text-sm font-medium disabled:opacity-50"
                >
                  {saving ? 'Saving...' : 'Save Notes'}
                </button>
              </div>
            )}
          </div>
        )}

        {/* Actions */}
        {isAssignedToMe && request.status !== 'COMPLETED' && (
          <div className="bg-white rounded-lg shadow-md p-6">
            <h2 className="text-lg font-semibold text-slate-900 mb-4">Actions</h2>
            <div className="flex gap-3">
              <button
                onClick={handleComplete}
                disabled={saving}
                className="flex-1 bg-green-600 hover:bg-green-700 text-white px-6 py-3 rounded-lg font-semibold disabled:opacity-50"
              >
                ✓ Mark as Completed
              </button>
              <button
                onClick={handleCancel}
                disabled={saving}
                className="px-6 py-3 border-2 border-red-600 text-red-600 hover:bg-red-50 rounded-lg font-semibold disabled:opacity-50"
              >
                Cancel Request
              </button>
            </div>
            <p className="text-sm text-slate-500 mt-3">
              Mark as completed once you've spoken with the client and helped them choose a mediator.
            </p>
          </div>
        )}

        {/* Completion Info */}
        {request.status === 'COMPLETED' && request.completedAt && (
          <div className="bg-green-50 border border-green-200 rounded-lg p-6">
            <div className="flex items-center text-green-700">
              <svg className="w-6 h-6 mr-2" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M9 12l2 2 4-4m6 2a9 9 0 11-18 0 9 9 0 0118 0z" />
              </svg>
              <span className="font-semibold">Request Completed</span>
            </div>
            <p className="text-green-600 mt-2">
              Completed {getTimeSince(request.completedAt)} by Officer #{request.assignedOfficerId?.slice(-4)}
            </p>
          </div>
        )}
      </div>
    </div>
  );
};

export default StaffIntakeDetail;
