import React, { useEffect, useState } from 'react';
import { useNavigate } from 'react-router-dom';
import { IntakeService } from '../../services/IntakeService';
import { IntakeRequest } from '../../types';

const StaffIntakeQueue: React.FC = () => {
  const navigate = useNavigate();
  const [requests, setRequests] = useState<IntakeRequest[]>([]);
  const [loading, setLoading] = useState(true);
  const [filter, setFilter] = useState<'all' | 'pending' | 'in_progress' | 'completed'>('pending');

  useEffect(() => {
    loadRequests();
  }, [filter]);

  const loadRequests = async () => {
    setLoading(true);
    const allRequests = await IntakeService.getPendingRequests();
    
    // Filter based on status
    const filtered = filter === 'all' 
      ? allRequests 
      : allRequests.filter(r => r.status.toLowerCase() === filter);
    
    setRequests(filtered);
    setLoading(false);
  };

  const getTimeSince = (timestamp: number): string => {
    const minutes = Math.floor((Date.now() - timestamp) / 60000);
    if (minutes < 60) return `${minutes} min ago`;
    const hours = Math.floor(minutes / 60);
    if (hours < 24) return `${hours} hour${hours > 1 ? 's' : ''} ago`;
    const days = Math.floor(hours / 24);
    return `${days} day${days > 1 ? 's' : ''} ago`;
  };

  const getUrgencyColor = (timestamp: number): string => {
    const hours = (Date.now() - timestamp) / (1000 * 60 * 60);
    if (hours < 1) return 'bg-green-100 text-green-800 border-green-200';
    if (hours < 4) return 'bg-yellow-100 text-yellow-800 border-yellow-200';
    return 'bg-red-100 text-red-800 border-red-200';
  };

  const getStatusBadge = (status: string) => {
    const styles = {
      PENDING: 'bg-yellow-100 text-yellow-800',
      IN_PROGRESS: 'bg-blue-100 text-blue-800',
      COMPLETED: 'bg-green-100 text-green-800',
      CANCELLED: 'bg-gray-100 text-gray-800'
    };
    return styles[status as keyof typeof styles] || styles.PENDING;
  };

  if (loading) {
    return (
      <div className="flex items-center justify-center min-h-screen">
        <div className="text-center">
          <div className="inline-block w-8 h-8 border-4 border-indigo-600 border-t-transparent rounded-full animate-spin"></div>
          <p className="mt-4 text-slate-600">Loading requests...</p>
        </div>
      </div>
    );
  }

  return (
    <div className="min-h-screen bg-gray-50 py-8">
      <div className="max-w-7xl mx-auto px-4 sm:px-6 lg:px-8">
        {/* Header */}
        <div className="mb-8">
          <h1 className="text-3xl font-bold text-slate-900">Intake Request Queue</h1>
          <p className="mt-2 text-slate-600">Manage client call-back requests</p>
        </div>

        {/* Stats Cards */}
        <div className="grid grid-cols-1 md:grid-cols-4 gap-6 mb-8">
          <div className="bg-white rounded-lg shadow p-6">
            <div className="text-sm font-medium text-slate-600">Total Pending</div>
            <div className="mt-2 text-3xl font-bold text-indigo-600">
              {requests.filter(r => r.status === 'PENDING').length}
            </div>
          </div>
          <div className="bg-white rounded-lg shadow p-6">
            <div className="text-sm font-medium text-slate-600">In Progress</div>
            <div className="mt-2 text-3xl font-bold text-blue-600">
              {requests.filter(r => r.status === 'IN_PROGRESS').length}
            </div>
          </div>
          <div className="bg-white rounded-lg shadow p-6">
            <div className="text-sm font-medium text-slate-600">Completed Today</div>
            <div className="mt-2 text-3xl font-bold text-green-600">
              {requests.filter(r => r.status === 'COMPLETED' && r.completedAt && Date.now() - r.completedAt < 86400000).length}
            </div>
          </div>
          <div className="bg-white rounded-lg shadow p-6">
            <div className="text-sm font-medium text-slate-600">⚠️ Over 4 Hours</div>
            <div className="mt-2 text-3xl font-bold text-red-600">
              {requests.filter(r => r.status === 'PENDING' && (Date.now() - r.createdAt) > (4 * 60 * 60 * 1000)).length}
            </div>
          </div>
        </div>

        {/* Filter Tabs */}
        <div className="bg-white rounded-lg shadow mb-6">
          <div className="border-b border-gray-200">
            <nav className="flex -mb-px">
              {['pending', 'in_progress', 'completed', 'all'].map((tab) => (
                <button
                  key={tab}
                  onClick={() => setFilter(tab as any)}
                  className={`py-4 px-6 text-sm font-medium border-b-2 capitalize ${
                    filter === tab
                      ? 'border-indigo-600 text-indigo-600'
                      : 'border-transparent text-gray-500 hover:text-gray-700 hover:border-gray-300'
                  }`}
                >
                  {tab.replace('_', ' ')}
                </button>
              ))}
            </nav>
          </div>
        </div>

        {/* Request List */}
        <div className="space-y-4">
          {requests.length === 0 ? (
            <div className="bg-white rounded-lg shadow p-12 text-center">
              <div className="text-gray-400 mb-4">
                <svg className="mx-auto h-12 w-12" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                  <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M20 13V6a2 2 0 00-2-2H6a2 2 0 00-2 2v7m16 0v5a2 2 0 01-2 2H6a2 2 0 01-2-2v-5m16 0h-2.586a1 1 0 00-.707.293l-2.414 2.414a1 1 0 01-.707.293h-3.172a1 1 0 01-.707-.293l-2.414-2.414A1 1 0 006.586 13H4" />
                </svg>
              </div>
              <h3 className="text-lg font-medium text-gray-900 mb-2">No requests found</h3>
              <p className="text-gray-500">
                {filter === 'pending' && "No pending requests at the moment."}
                {filter === 'in_progress' && "No requests currently in progress."}
                {filter === 'completed' && "No completed requests."}
                {filter === 'all' && "No intake requests yet."}
              </p>
            </div>
          ) : (
            requests.map((request) => (
              <div
                key={request.id}
                className="bg-white rounded-lg shadow hover:shadow-md transition-shadow cursor-pointer"
                onClick={() => navigate(`/staff/intake/${request.id}`)}
              >
                <div className="p-6">
                  <div className="flex items-start justify-between">
                    <div className="flex-1">
                      {/* Header with name and time */}
                      <div className="flex items-center gap-3 mb-3">
                        <h3 className="text-lg font-semibold text-slate-900">
                          {request.clientName}
                        </h3>
                        <span className={`px-3 py-1 rounded-full text-xs font-medium ${getStatusBadge(request.status)}`}>
                          {request.status.replace('_', ' ')}
                        </span>
                        {request.status === 'PENDING' && (
                          <span className={`px-3 py-1 rounded-full text-xs font-medium border ${getUrgencyColor(request.createdAt)}`}>
                            {getTimeSince(request.createdAt)}
                          </span>
                        )}
                      </div>

                      {/* Contact Info */}
                      <div className="grid grid-cols-1 md:grid-cols-2 gap-3 mb-4">
                        <div className="flex items-center text-sm text-slate-600">
                          <svg className="w-4 h-4 mr-2" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                            <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M3 8l7.89 5.26a2 2 0 002.22 0L21 8M5 19h14a2 2 0 002-2V7a2 2 0 00-2-2H5a2 2 0 00-2 2v10a2 2 0 002 2z" />
                          </svg>
                          {request.clientEmail}
                        </div>
                        <div className="flex items-center text-sm text-slate-600">
                          <svg className="w-4 h-4 mr-2" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                            <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M3 5a2 2 0 012-2h3.28a1 1 0 01.948.684l1.498 4.493a1 1 0 01-.502 1.21l-2.257 1.13a11.042 11.042 0 005.516 5.516l1.13-2.257a1 1 0 011.21-.502l4.493 1.498a1 1 0 01.684.949V19a2 2 0 01-2 2h-1C9.716 21 3 14.284 3 6V5z" />
                          </svg>
                          {request.clientPhone}
                        </div>
                      </div>

                      {/* Description */}
                      {request.description && (
                        <p className="text-sm text-slate-600 line-clamp-2 mb-3">
                          {request.description}
                        </p>
                      )}

                      {/* Assigned Officer */}
                      {request.assignedOfficerId && (
                        <div className="flex items-center text-sm text-slate-500 mb-2">
                          <svg className="w-4 h-4 mr-2" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                            <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M16 7a4 4 0 11-8 0 4 4 0 018 0zM12 14a7 7 0 00-7 7h14a7 7 0 00-7-7z" />
                          </svg>
                          Assigned to Officer #{request.assignedOfficerId.slice(-4)}
                        </div>
                      )}

                      {/* Notes Preview */}
                      {request.officerNotes && (
                        <div className="bg-slate-50 rounded p-3 mt-3">
                          <div className="text-xs font-medium text-slate-500 mb-1">Officer Notes:</div>
                          <p className="text-sm text-slate-700 line-clamp-1">{request.officerNotes}</p>
                        </div>
                      )}
                    </div>

                    {/* Action Arrow */}
                    <div className="ml-4">
                      <svg className="w-6 h-6 text-slate-400" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                        <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M9 5l7 7-7 7" />
                      </svg>
                    </div>
                  </div>
                </div>
              </div>
            ))
          )}
        </div>
      </div>
    </div>
  );
};

export default StaffIntakeQueue;
