/**
 * Simple Email API Server
 * Handles SendGrid email sending (server-side only)
 */

import express from 'express';
import cors from 'cors';
import sgMail from '@sendgrid/mail';
import Stripe from 'stripe';
import dotenv from 'dotenv';

// Load environment variables
dotenv.config();

const app = express();
const PORT = process.env.PORT || 3001;
const stripe = new Stripe(process.env.STRIPE_SECRET_KEY);

// Configure SendGrid
sgMail.setApiKey(process.env.SENDGRID_API_KEY);

// Middleware
app.use(cors());
app.use(express.json());

// Health check endpoint
app.get('/health', (req, res) => {
  res.json({ status: 'ok', service: 'medi8-email-api' });
});

// Send email endpoint
app.post('/api/send-email', async (req, res) => {
  try {
    const { to, from, replyTo, subject, text, html } = req.body;

    // Validate required fields
    if (!to || !subject || (!text && !html)) {
      return res.status(400).json({ 
        error: 'Missing required fields: to, subject, and text/html' 
      });
    }

    // Prepare email
    const msg = {
      to,
      from: from || {
        email: process.env.FROM_EMAIL || 'hello@medi8.com.au',
        name: process.env.FROM_NAME || 'medi8 Team'
      },
      replyTo: replyTo || process.env.REPLY_TO_EMAIL || 'support@medi8.com.au',
      subject,
      text,
      html
    };

    // Send via SendGrid
    await sgMail.send(msg);

    console.log('✅ Email sent successfully to:', to);
    
    res.json({ 
      success: true, 
      message: 'Email sent successfully',
      to 
    });

  } catch (error) {
    console.error('❌ Email sending error:', error);
    
    res.status(500).json({ 
      success: false,
      error: 'Failed to send email',
      details: error.message 
    });
  }
});

// Create Stripe Checkout Session
app.post('/api/create-checkout-session', async (req, res) => {
  try {
    const { 
      mediatorId, 
      mediatorName, 
      clientName, 
      clientEmail, 
      amount, 
      description, 
      metadata 
    } = req.body;

    // Validate required fields
    if (!amount || !clientEmail) {
      return res.status(400).json({ 
        error: 'Missing required fields: amount, clientEmail' 
      });
    }

    // Calculate platform fee (10%)
    const platformFee = Math.round(amount * 0.10);

    // Create Stripe Checkout Session
    const session = await stripe.checkout.sessions.create({
      mode: 'payment',
      payment_method_types: ['card'],
      customer_email: clientEmail,
      line_items: [
        {
          price_data: {
            currency: 'aud',
            product_data: {
              name: description || 'Pre-Mediation Session',
              description: `Mediator: ${mediatorName || 'Professional Mediator'}`,
            },
            unit_amount: amount, // amount in cents
          },
          quantity: 1,
        },
      ],
      // For future: Add destination charges with mediator's Stripe account
      // payment_intent_data: {
      //   application_fee_amount: platformFee,
      //   transfer_data: {
      //     destination: mediatorStripeAccountId,
      //   },
      // },
      metadata: {
        mediatorId: mediatorId || 'unknown',
        mediatorName: mediatorName || 'Unknown',
        clientName: clientName || 'Unknown',
        clientEmail: clientEmail,
        platformFee: platformFee.toString(),
        ...metadata
      },
      success_url: `${process.env.APP_URL || 'http://localhost:3000'}/#/booking-success?session_id={CHECKOUT_SESSION_ID}`,
      cancel_url: `${process.env.APP_URL || 'http://localhost:3000'}/#/mediators`,
    });

    console.log('✅ Checkout session created:', session.id);
    
    res.json({ 
      success: true,
      sessionId: session.id,
      url: session.url
    });

  } catch (error) {
    console.error('❌ Stripe checkout error:', error);
    
    res.status(500).json({ 
      success: false,
      error: 'Failed to create checkout session',
      details: error.message 
    });
  }
});

// Verify payment and send confirmation emails
app.post('/api/verify-payment', async (req, res) => {
  try {
    const { sessionId } = req.body;

    if (!sessionId) {
      return res.status(400).json({ error: 'Missing session ID' });
    }

    // Retrieve the session from Stripe
    const session = await stripe.checkout.sessions.retrieve(sessionId);

    if (session.payment_status !== 'paid') {
      return res.status(400).json({ error: 'Payment not completed' });
    }

    // Extract metadata
    const { mediatorName, clientName, clientEmail, mediatorId } = session.metadata;
    const bookingId = session.metadata.bookingId || 'bk-' + sessionId.slice(-8);

    // Send client confirmation email
    const clientEmailMsg = {
      to: clientEmail,
      from: {
        email: process.env.FROM_EMAIL || 'hello@medi8.com.au',
        name: process.env.FROM_NAME || 'medi8 Team'
      },
      replyTo: process.env.REPLY_TO_EMAIL || 'support@medi8.com.au',
      subject: `Booking Confirmed - ${mediatorName} will contact you soon`,
      text: `Thank you, ${clientName}!\n\nYour pre-mediation booking with ${mediatorName} has been confirmed.\n\nWhat happens next:\n1. ${mediatorName} will contact you within 48 hours\n2. They'll send you the Agreement to Mediate to review\n3. You'll schedule your pre-mediation session together\n\nYour booking reference: ${bookingId}\n\nmedi8 Team`,
      html: `
        <div style="font-family: Arial, sans-serif; max-width: 600px; margin: 0 auto;">
          <h2 style="color: #111827;">Thank you, ${clientName}</h2>
          <p style="color: #374151;">Your pre-mediation booking with <strong>${mediatorName}</strong> has been confirmed.</p>
          
          <h3 style="color: #111827; margin-top: 24px;">What happens next:</h3>
          <ol style="color: #374151; line-height: 1.8;">
            <li>${mediatorName} will contact you within 48 hours</li>
            <li>They'll send you the Agreement to Mediate to review</li>
            <li>You'll schedule your pre-mediation session together</li>
          </ol>
          
          <div style="background-color: #f3f4f6; border-left: 4px solid #4f46e5; padding: 16px; margin: 24px 0;">
            <p style="margin: 0; color: #374151;">
              <strong>Your booking reference:</strong> ${bookingId}
            </p>
          </div>
          
          <p style="color: #6b7280; font-size: 12px; margin-top: 32px;">
            Questions? Contact us at support@medi8.com.au
          </p>
        </div>
      `
    };

    await sgMail.send(clientEmailMsg);
    console.log('✅ Client confirmation email sent to:', clientEmail);

    res.json({ 
      success: true,
      bookingId,
      emailSent: true
    });

  } catch (error) {
    console.error('❌ Payment verification error:', error);
    res.status(500).json({ 
      success: false,
      error: 'Failed to verify payment',
      details: error.message 
    });
  }
});

// Start server
app.listen(PORT, () => {
  console.log(`🚀 Email & Payment API running on port ${PORT}`);
  console.log(`📧 SendGrid configured: ${process.env.SENDGRID_API_KEY ? 'Yes' : 'No'}`);
  console.log(`💳 Stripe configured: ${process.env.STRIPE_SECRET_KEY ? 'Yes' : 'No'}`);
  console.log(`📨 From email: ${process.env.FROM_EMAIL || 'hello@medi8.com.au'}`);
});
