import { DatabaseService } from './DatabaseService';
import { AvailabilityPeriod, AvailabilityStatus, MediatorAvailability } from '../types';

export class AvailabilityService {
  /**
   * Create a new unavailability period
   */
  static async createUnavailablePeriod(
    mediatorId: string,
    startDate: number,
    endDate: number,
    reason?: string,
    notes?: string
  ): Promise<AvailabilityPeriod> {
    const period: AvailabilityPeriod = {
      id: 'avail-' + Math.random().toString(36).substr(2, 9),
      mediatorId,
      startDate,
      endDate,
      reason,
      notes,
      createdAt: Date.now(),
      updatedAt: Date.now()
    };

    await DatabaseService.put('availabilityPeriods', period);
    return period;
  }

  /**
   * Get all availability periods for a mediator
   */
  static async getMediatorPeriods(mediatorId: string): Promise<AvailabilityPeriod[]> {
    const all = await DatabaseService.getAll<AvailabilityPeriod>('availabilityPeriods');
    return all
      .filter(p => p.mediatorId === mediatorId)
      .sort((a, b) => a.startDate - b.startDate);
  }

  /**
   * Get current availability status for a mediator
   */
  static async getMediatorAvailability(mediatorId: string): Promise<MediatorAvailability> {
    const periods = await this.getMediatorPeriods(mediatorId);
    const now = Date.now();

    // Find current unavailability period (if any)
    const currentPeriod = periods.find(
      p => p.startDate <= now && p.endDate >= now
    );

    // Find upcoming periods
    const upcomingPeriods = periods.filter(
      p => p.startDate > now
    );

    return {
      mediatorId,
      status: currentPeriod ? AvailabilityStatus.UNAVAILABLE : AvailabilityStatus.AVAILABLE,
      currentPeriod,
      upcomingPeriods,
      lastUpdated: Date.now()
    };
  }

  /**
   * Check if a mediator is currently available
   */
  static async isAvailable(mediatorId: string): Promise<boolean> {
    const availability = await this.getMediatorAvailability(mediatorId);
    return availability.status === AvailabilityStatus.AVAILABLE;
  }

  /**
   * Update an existing period
   */
  static async updatePeriod(
    periodId: string,
    updates: Partial<AvailabilityPeriod>
  ): Promise<void> {
    const period = await DatabaseService.getById<AvailabilityPeriod>('availabilityPeriods', periodId);
    if (period) {
      Object.assign(period, updates);
      period.updatedAt = Date.now();
      await DatabaseService.put('availabilityPeriods', period);
    }
  }

  /**
   * Delete a period
   */
  static async deletePeriod(periodId: string): Promise<void> {
    await DatabaseService.delete('availabilityPeriods', periodId);
  }

  /**
   * Get all mediators who are currently unavailable
   */
  static async getUnavailableMediators(): Promise<string[]> {
    const all = await DatabaseService.getAll<AvailabilityPeriod>('availabilityPeriods');
    const now = Date.now();

    const unavailableMediatorIds = new Set<string>();
    
    all.forEach(period => {
      if (period.startDate <= now && period.endDate >= now) {
        unavailableMediatorIds.add(period.mediatorId);
      }
    });

    return Array.from(unavailableMediatorIds);
  }

  /**
   * Clean up expired periods (optional maintenance)
   */
  static async cleanupExpiredPeriods(): Promise<number> {
    const all = await DatabaseService.getAll<AvailabilityPeriod>('availabilityPeriods');
    const now = Date.now();
    const oneMonthAgo = now - (30 * 24 * 60 * 60 * 1000);

    let deleted = 0;
    for (const period of all) {
      // Delete periods that ended more than a month ago
      if (period.endDate < oneMonthAgo) {
        await this.deletePeriod(period.id);
        deleted++;
      }
    }

    return deleted;
  }

  /**
   * Format date for display
   */
  static formatDate(timestamp: number): string {
    return new Date(timestamp).toLocaleDateString('en-AU', {
      day: 'numeric',
      month: 'short',
      year: 'numeric'
    });
  }

  /**
   * Format date range for display
   */
  static formatDateRange(startDate: number, endDate: number): string {
    const start = this.formatDate(startDate);
    const end = this.formatDate(endDate);
    
    if (start === end) {
      return start;
    }
    
    return `${start} - ${end}`;
  }
}
