
import { DatabaseService } from './DatabaseService';
import { BookingRequest, BookingStatus, PaymentEvent, PaymentKind } from '../types';

export class BookingService {
  static async createRequest(request: BookingRequest): Promise<void> {
    await DatabaseService.put('bookings', request);
  }

  static async getById(id: string): Promise<BookingRequest | undefined> {
    return DatabaseService.getById<BookingRequest>('bookings', id);
  }

  static async getByMediator(mediatorId: string): Promise<BookingRequest[]> {
    const all = await DatabaseService.getAll<BookingRequest>('bookings');
    return all.filter(b => b.mediatorId === mediatorId);
  }

  static async updateStatus(bookingId: string, status: BookingStatus, feeTotalCents?: number): Promise<void> {
    const booking = await DatabaseService.getById<BookingRequest>('bookings', bookingId);
    if (booking) {
      booking.status = status;
      if (feeTotalCents !== undefined) booking.feeTotalCents = feeTotalCents;
      await DatabaseService.put('bookings', booking);
    }
  }

  static async recordPayment(payment: PaymentEvent): Promise<void> {
    await DatabaseService.put('payments', payment);
    
    // Auto-update status based on payment
    const booking = await DatabaseService.getById<BookingRequest>('bookings', payment.bookingId);
    if (booking) {
      if (payment.kind === PaymentKind.DEPOSIT) {
        booking.status = BookingStatus.DEPOSIT_PAID;
      } else if (payment.kind === PaymentKind.BALANCE) {
        booking.status = BookingStatus.BALANCE_PAID;
      }
      await DatabaseService.put('bookings', booking);
    }
  }

  static async getPaymentsForBooking(bookingId: string): Promise<PaymentEvent[]> {
    const all = await DatabaseService.getAll<PaymentEvent>('payments');
    return all.filter(p => p.bookingId === bookingId);
  }
}
