
import { openDB, IDBPDatabase } from 'idb';

const DB_NAME = 'medi8_db';
const DB_VERSION = 5; // Incremented for operating hours system

export class DatabaseService {
  private static db: IDBPDatabase | null = null;

  static async getDB() {
    if (this.db) return this.db;

    this.db = await openDB(DB_NAME, DB_VERSION, {
      upgrade(db) {
        if (!db.objectStoreNames.contains('users')) db.createObjectStore('users', { keyPath: 'id' });
        if (!db.objectStoreNames.contains('mediators')) db.createObjectStore('mediators', { keyPath: 'id' });
        if (!db.objectStoreNames.contains('enquiries')) db.createObjectStore('enquiries', { keyPath: 'id' });
        if (!db.objectStoreNames.contains('bookings')) db.createObjectStore('bookings', { keyPath: 'id' });
        if (!db.objectStoreNames.contains('payments')) db.createObjectStore('payments', { keyPath: 'id' });
        if (!db.objectStoreNames.contains('categories')) db.createObjectStore('categories', { keyPath: 'id' });
        if (!db.objectStoreNames.contains('audit')) db.createObjectStore('audit', { keyPath: 'id' });
        if (!db.objectStoreNames.contains('fdr_cases')) db.createObjectStore('fdr_cases', { keyPath: 'id' });
        
        // Phase 1: New engagement workflow stores
        if (!db.objectStoreNames.contains('engagementForms')) db.createObjectStore('engagementForms', { keyPath: 'id' });
        if (!db.objectStoreNames.contains('abandonedEngagements')) db.createObjectStore('abandonedEngagements', { keyPath: 'id' });
        if (!db.objectStoreNames.contains('intakeRequests')) db.createObjectStore('intakeRequests', { keyPath: 'id' });
        
        // Phase 2b: Availability management
        if (!db.objectStoreNames.contains('availabilityPeriods')) db.createObjectStore('availabilityPeriods', { keyPath: 'id' });
        if (!db.objectStoreNames.contains('operatingHours')) db.createObjectStore('operatingHours', { keyPath: 'id' });
      },
    });

    return this.db;
  }

  static async getAll<T>(storeName: string): Promise<T[]> {
    const db = await this.getDB();
    return db.getAll(storeName);
  }

  static async getById<T>(storeName: string, id: string): Promise<T | undefined> {
    const db = await this.getDB();
    return db.get(storeName, id);
  }

  static async put<T>(storeName: string, item: T): Promise<void> {
    const db = await this.getDB();
    await db.put(storeName, item);
  }

  static async delete(storeName: string, id: string): Promise<void> {
    const db = await this.getDB();
    await db.delete(storeName, id);
  }

  static async clearAll() {
    const db = await this.getDB();
    const stores = [
      'users', 'mediators', 'enquiries', 'bookings', 'payments', 
      'categories', 'audit', 'fdr_cases',
      'engagementForms', 'abandonedEngagements', 'intakeRequests'
    ];
    for (const store of stores) {
      if (db.objectStoreNames.contains(store as any)) await db.clear(store as any);
    }
  }
}
