/**
 * EmailService - Real Email Sending via SendGrid
 * Phase 2a: Production-ready email service
 */

export interface EmailTemplate {
  to: string;
  subject: string;
  htmlBody: string;
  textBody: string;
}

export class EmailService {
  private static FROM_EMAIL = import.meta.env.VITE_FROM_EMAIL || 'hello@medi8.com.au';
  private static FROM_NAME = import.meta.env.VITE_FROM_NAME || 'medi8 Team';
  private static REPLY_TO = import.meta.env.VITE_REPLY_TO_EMAIL || 'support@medi8.com.au';
  private static BUSINESS_NAME = import.meta.env.VITE_BUSINESS_NAME || 'Mediation Institute Pty Ltd';
  private static BUSINESS_ABN = import.meta.env.VITE_BUSINESS_ABN || 'ABN 43 166 175 962';
  private static BUSINESS_ADDRESS = import.meta.env.VITE_BUSINESS_ADDRESS || '268-278 National Park Road, Loch Sport, VIC 3851';

  /**
   * Get email footer with branding
   */
  private static getEmailFooter(): string {
    return `
      <div style="margin-top: 40px; padding-top: 20px; border-top: 1px solid #e5e7eb; color: #6b7280; font-size: 12px;">
        <p style="margin: 0 0 8px 0;"><strong>${this.FROM_NAME}</strong></p>
        <p style="margin: 0 0 4px 0; font-size: 11px;">${this.BUSINESS_NAME}</p>
        <p style="margin: 0 0 4px 0; font-size: 11px;">${this.BUSINESS_ABN}</p>
        <p style="margin: 0 0 16px 0; font-size: 11px;">${this.BUSINESS_ADDRESS}</p>
        <p style="margin: 0; font-size: 11px;">
          Questions? Contact us at <a href="mailto:${this.REPLY_TO}" style="color: #4f46e5;">${this.REPLY_TO}</a>
        </p>
      </div>
    `;
  }

  /**
   * Wrap email content in professional HTML template
   */
  private static wrapInTemplate(content: string, subject: string): string {
    return `
<!DOCTYPE html>
<html>
<head>
  <meta charset="UTF-8">
  <meta name="viewport" content="width=device-width, initial-scale=1.0">
  <title>${subject}</title>
</head>
<body style="margin: 0; padding: 0; font-family: -apple-system, BlinkMacSystemFont, 'Segoe UI', Roboto, 'Helvetica Neue', Arial, sans-serif; background-color: #f9fafb;">
  <div style="max-width: 600px; margin: 0 auto; background-color: #ffffff;">
    <!-- Header -->
    <div style="background: linear-gradient(135deg, #4f46e5 0%, #6366f1 100%); padding: 32px; text-align: center;">
      <h1 style="color: #ffffff; margin: 0; font-size: 28px; font-weight: 700;">${this.FROM_NAME}</h1>
      <p style="color: #e0e7ff; margin: 8px 0 0 0; font-size: 14px;">Professional Mediator Marketplace</p>
    </div>
    
    <!-- Content -->
    <div style="padding: 32px;">
      ${content}
      ${this.getEmailFooter()}
    </div>
  </div>
</body>
</html>
    `;
  }

  /**
   * Send email - logs to console in dev mode, sends via API in production
   */
  private static async sendEmail(email: EmailTemplate): Promise<void> {
    const wrappedHtml = this.wrapInTemplate(email.htmlBody, email.subject);

    // In development, log to console
    if (import.meta.env.DEV) {
      console.log('📧 EMAIL SENT (SendGrid-Ready)');
      console.log('To:', email.to);
      console.log('From:', `${this.FROM_NAME} <${this.FROM_EMAIL}>`);
      console.log('Reply-To:', this.REPLY_TO);
      console.log('Subject:', email.subject);
      console.log('---');
      console.log('✉️ Email is ready to send via SendGrid');
      console.log('✉️ In production, this will be sent to:', email.to);
      console.log('---');
    }

    // Call backend API to send email
    try {
      const apiUrl = import.meta.env.VITE_EMAIL_API_URL || 'http://localhost:3001';
      
      const response = await fetch(`${apiUrl}/api/send-email`, {
        method: 'POST',
        headers: {
          'Content-Type': 'application/json',
        },
        body: JSON.stringify({
          to: email.to,
          from: {
            email: this.FROM_EMAIL,
            name: this.FROM_NAME
          },
          replyTo: this.REPLY_TO,
          subject: email.subject,
          text: email.textBody,
          html: wrappedHtml
        })
      });

      if (!response.ok) {
        const error = await response.json();
        throw new Error(error.details || 'Email sending failed');
      }

      const result = await response.json();
      console.log('✅ Real email sent via SendGrid:', result);
      
    } catch (error) {
      console.error('❌ Email API error:', error);
      // In development, don't throw - just log
      if (!import.meta.env.DEV) {
        throw error;
      }
    }
    
    return Promise.resolve();
  }

  /**
   * Send confirmation email to client
   */
  static async sendClientConfirmation(
    clientEmail: string,
    clientName: string,
    mediatorName: string,
    referenceNumber: string
  ): Promise<void> {
    const email: EmailTemplate = {
      to: clientEmail,
      subject: `Booking Confirmed - ${mediatorName} will contact you soon`,
      htmlBody: `
        <h2 style="color: #111827; margin: 0 0 16px 0;">Thank you, ${clientName}</h2>
        <p style="color: #374151; line-height: 1.6; margin: 0 0 24px 0;">
          Your pre-mediation booking with <strong>${mediatorName}</strong> has been confirmed.
        </p>
        
        <h3 style="color: #111827; margin: 24px 0 12px 0; font-size: 18px;">What happens next:</h3>
        <ol style="color: #374151; line-height: 1.8; margin: 0 0 24px 0; padding-left: 24px;">
          <li>${mediatorName} will contact you within 48 hours</li>
          <li>They'll send you the Agreement to Mediate to review</li>
          <li>You'll schedule your pre-mediation session together</li>
        </ol>
        
        <div style="background-color: #f3f4f6; border-left: 4px solid #4f46e5; padding: 16px; margin: 24px 0;">
          <p style="margin: 0; color: #374151;">
            <strong style="color: #111827;">Your booking reference:</strong> ${referenceNumber}
          </p>
        </div>
      `,
      textBody: `
Thank you, ${clientName}

Your pre-mediation booking with ${mediatorName} has been confirmed.

What happens next:
1. ${mediatorName} will contact you within 48 hours
2. They'll send you the Agreement to Mediate to review
3. You'll schedule your pre-mediation session together

Your booking reference: ${referenceNumber}

Questions? Contact our support team at ${this.REPLY_TO}

${this.FROM_NAME}
${this.BUSINESS_NAME}
${this.BUSINESS_ABN}
      `
    };

    await this.sendEmail(email);
  }

  /**
   * Send notification to mediator
   */
  static async sendMediatorNotification(
    mediatorEmail: string,
    mediatorName: string,
    clientName: string,
    clientEmail: string,
    clientPhone: string,
    disputeType: string,
    bookingId: string
  ): Promise<void> {
    const email: EmailTemplate = {
      to: mediatorEmail,
      subject: 'New Booking - Action Required Within 48 Hours',
      htmlBody: `
        <h2 style="color: #111827; margin: 0 0 16px 0;">New Pre-Mediation Booking</h2>
        <p style="color: #374151; line-height: 1.6; margin: 0 0 24px 0;">
          You have a new booking request that requires response within 48 hours.
        </p>
        
        <div style="background-color: #f3f4f6; padding: 20px; border-radius: 8px; margin: 24px 0;">
          <h3 style="color: #111827; margin: 0 0 12px 0; font-size: 16px;">Client Details:</h3>
          <table style="width: 100%; color: #374151; line-height: 1.8;">
            <tr>
              <td style="font-weight: 600; width: 120px;">Name:</td>
              <td>${clientName}</td>
            </tr>
            <tr>
              <td style="font-weight: 600;">Email:</td>
              <td><a href="mailto:${clientEmail}" style="color: #4f46e5;">${clientEmail}</a></td>
            </tr>
            <tr>
              <td style="font-weight: 600;">Phone:</td>
              <td><a href="tel:${clientPhone}" style="color: #4f46e5;">${clientPhone}</a></td>
            </tr>
            <tr>
              <td style="font-weight: 600;">Dispute Type:</td>
              <td>${disputeType}</td>
            </tr>
          </table>
        </div>
        
        <h3 style="color: #111827; margin: 24px 0 12px 0; font-size: 18px;">Next Steps (within 48 hours):</h3>
        <ol style="color: #374151; line-height: 1.8; margin: 0 0 24px 0; padding-left: 24px;">
          <li>Send Agreement to Mediate to the client</li>
          <li>Schedule the pre-mediation session</li>
          <li>Mark as "Scheduled" in your dashboard</li>
        </ol>
      `,
      textBody: `
New Pre-Mediation Booking

You have a new booking request that requires response within 48 hours.

Client Details:
- Name: ${clientName}
- Email: ${clientEmail}
- Phone: ${clientPhone}
- Dispute Type: ${disputeType}

Next Steps (within 48 hours):
1. Send Agreement to Mediate to the client
2. Schedule the pre-mediation session
3. Mark as "Scheduled" in your dashboard

${this.FROM_NAME}
${this.BUSINESS_NAME}
      `
    };

    await this.sendEmail(email);
  }

  /**
   * Send abandoned cart follow-up emails
   */
  static async sendAbandonedFollowUp1(
    clientEmail: string,
    clientName: string,
    mediatorName: string,
    formId: string
  ): Promise<void> {
    const email: EmailTemplate = {
      to: clientEmail,
      subject: 'Did you need help completing your booking?',
      htmlBody: `
        <h2 style="color: #111827; margin: 0 0 16px 0;">Hi ${clientName},</h2>
        <p style="color: #374151; line-height: 1.6; margin: 0 0 24px 0;">
          We noticed you started booking <strong>${mediatorName}</strong> but didn't finish.
        </p>
        
        <div style="text-align: center; margin: 32px 0;">
          <a href="${import.meta.env.VITE_APP_URL}/#/engagement/${formId}" style="display: inline-block; background-color: #4f46e5; color: #ffffff; text-decoration: none; padding: 12px 32px; border-radius: 6px; font-weight: 600;">
            Continue Your Booking
          </a>
        </div>
        
        <p style="color: #6b7280; font-size: 14px; line-height: 1.6; margin: 24px 0 0 0;">
          Need help? Call us at ${import.meta.env.VITE_SUPPORT_PHONE || '0401 293 500'}
        </p>
      `,
      textBody: `
Hi ${clientName},

We noticed you started booking ${mediatorName} but didn't finish.

Continue your booking here: ${import.meta.env.VITE_APP_URL}/#/engagement/${formId}

Need help? Call us at ${import.meta.env.VITE_SUPPORT_PHONE || '0401 293 500'}

${this.FROM_NAME}
      `
    };

    await this.sendEmail(email);
  }

  static async sendIntakeRequestNotification(
    staffEmail: string,
    clientName: string,
    clientEmail: string,
    clientPhone: string,
    requestId: string
  ): Promise<void> {
    const email: EmailTemplate = {
      to: staffEmail,
      subject: 'New Call Back Request',
      htmlBody: `
        <h2 style="color: #111827; margin: 0 0 16px 0;">New Call Back Request</h2>
        
        <div style="background-color: #fef3c7; border-left: 4px solid #f59e0b; padding: 16px; margin: 24px 0;">
          <p style="margin: 0; color: #92400e; font-weight: 600;">
            ⏰ Please contact within 4 business hours
          </p>
        </div>
        
        <div style="background-color: #f3f4f6; padding: 20px; border-radius: 8px; margin: 24px 0;">
          <h3 style="color: #111827; margin: 0 0 12px 0; font-size: 16px;">Client Details:</h3>
          <table style="width: 100%; color: #374151; line-height: 1.8;">
            <tr>
              <td style="font-weight: 600; width: 120px;">Name:</td>
              <td>${clientName}</td>
            </tr>
            <tr>
              <td style="font-weight: 600;">Email:</td>
              <td><a href="mailto:${clientEmail}" style="color: #4f46e5;">${clientEmail}</a></td>
            </tr>
            <tr>
              <td style="font-weight: 600;">Phone:</td>
              <td><a href="tel:${clientPhone}" style="color: #4f46e5;">${clientPhone}</a></td>
            </tr>
          </table>
        </div>
      `,
      textBody: `
New Call Back Request

⏰ Please contact within 4 business hours

Client Details:
- Name: ${clientName}
- Email: ${clientEmail}
- Phone: ${clientPhone}

${this.FROM_NAME}
      `
    };

    await this.sendEmail(email);
  }
}
