import { DatabaseService } from './DatabaseService';
import {
  EngagementForm,
  EngagementStage,
  DisputeType,
  AbandonedEngagement,
  IntakeRequest
} from '../types';

export class EngagementService {
  /**
   * Create a new engagement form
   */
  static async createForm(mediatorId: string, estimatedFeeCents: number): Promise<EngagementForm> {
    const form: EngagementForm = {
      id: 'eng-' + Math.random().toString(36).substr(2, 9),
      mediatorId,
      stage: EngagementStage.CONTACT_INFO,
      
      // Contact Info (empty)
      clientName: '',
      clientEmail: '',
      clientPhone: '',
      preferredLocation: '',
      preferInPerson: false,
      
      // Dispute Details (empty)
      disputeType: '' as any,
      disputeDuration: '',
      otherPartyName: '',
      disputeDescription: '',
      
      // Legal Context (empty)
      hasCourtOrders: false,
      hasSafetyOrders: false,
      hasContracts: false,
      contextDetails: '',
      
      // Special Considerations (empty)
      needsDisabilityAccommodation: false,
      culturalOrLanguageNeeds: '',
      safetyConcerns: '',
      
      // Metadata
      createdAt: Date.now(),
      updatedAt: Date.now(),
      lastActivityAt: Date.now(),
      
      // Tracking
      estimatedFeeCents
    };
    
    await DatabaseService.put('engagementForms', form);
    return form;
  }

  /**
   * Update an engagement form (auto-save)
   */
  static async updateForm(id: string, updates: Partial<EngagementForm>): Promise<void> {
    const form = await DatabaseService.getById<EngagementForm>('engagementForms', id);
    if (form) {
      Object.assign(form, updates, {
        updatedAt: Date.now(),
        lastActivityAt: Date.now()
      });
      await DatabaseService.put('engagementForms', form);
    }
  }

  /**
   * Get form by ID
   */
  static async getById(id: string): Promise<EngagementForm | undefined> {
    return DatabaseService.getById<EngagementForm>('engagementForms', id);
  }

  /**
   * Complete the form and mark as ready for payment
   */
  static async completeForm(id: string): Promise<void> {
    const form = await this.getById(id);
    if (form) {
      form.stage = EngagementStage.PAYMENT;
      form.updatedAt = Date.now();
      await DatabaseService.put('engagementForms', form);
    }
  }

  /**
   * Mark form as paid and completed
   */
  static async markAsPaid(id: string): Promise<void> {
    const form = await this.getById(id);
    if (form) {
      form.stage = EngagementStage.COMPLETED;
      form.completedAt = Date.now();
      form.updatedAt = Date.now();
      await DatabaseService.put('engagementForms', form);
    }
  }

  /**
   * Get all engagement forms for a mediator
   */
  static async getByMediator(mediatorId: string): Promise<EngagementForm[]> {
    const all = await DatabaseService.getAll<EngagementForm>('engagementForms');
    return all.filter(f => f.mediatorId === mediatorId);
  }

  /**
   * Get all abandoned forms (not completed, inactive for > 1 hour)
   */
  static async getAbandonedForms(): Promise<EngagementForm[]> {
    const all = await DatabaseService.getAll<EngagementForm>('engagementForms');
    const oneHourAgo = Date.now() - (60 * 60 * 1000);
    
    return all.filter(f => 
      f.stage !== EngagementStage.COMPLETED && 
      f.stage !== EngagementStage.ABANDONED &&
      f.lastActivityAt < oneHourAgo &&
      f.clientEmail // Must have at least an email
    );
  }

  /**
   * Mark a form as abandoned
   */
  static async markAsAbandoned(formId: string): Promise<void> {
    const form = await this.getById(formId);
    if (form) {
      form.stage = EngagementStage.ABANDONED;
      form.updatedAt = Date.now();
      await DatabaseService.put('engagementForms', form);

      // Create abandoned engagement record for tracking
      const abandoned: AbandonedEngagement = {
        id: 'abn-' + Math.random().toString(36).substr(2, 9),
        formId: form.id,
        clientEmail: form.clientEmail,
        clientName: form.clientName,
        mediatorId: form.mediatorId,
        lastStage: form.stage,
        lastActivityAt: form.lastActivityAt,
        followUpsSent: 0,
        recovered: false
      };
      
      await DatabaseService.put('abandonedEngagements', abandoned);
    }
  }

  /**
   * Record that a follow-up was sent
   */
  static async recordFollowUp(formId: string): Promise<void> {
    const abandoned = await this.getAbandonedRecordByFormId(formId);
    if (abandoned) {
      abandoned.followUpsSent += 1;
      abandoned.lastFollowUpAt = Date.now();
      await DatabaseService.put('abandonedEngagements', abandoned);
    }
  }

  /**
   * Mark an abandoned engagement as recovered
   */
  static async markAsRecovered(formId: string): Promise<void> {
    const abandoned = await this.getAbandonedRecordByFormId(formId);
    if (abandoned) {
      abandoned.recovered = true;
      abandoned.recoveredAt = Date.now();
      await DatabaseService.put('abandonedEngagements', abandoned);
    }
  }

  /**
   * Get abandoned engagement record by form ID
   */
  static async getAbandonedRecordByFormId(formId: string): Promise<AbandonedEngagement | undefined> {
    const all = await DatabaseService.getAll<AbandonedEngagement>('abandonedEngagements');
    return all.find(a => a.formId === formId);
  }

  /**
   * Get all abandoned engagements
   */
  static async getAllAbandoned(): Promise<AbandonedEngagement[]> {
    return DatabaseService.getAll<AbandonedEngagement>('abandonedEngagements');
  }
}
