
import { DatabaseService } from './DatabaseService';
import { FDRCase, FDRStage, ScreeningOutcome } from '../types';

export class FDRService {
  static async createCase(c: FDRCase): Promise<void> {
    await DatabaseService.put('fdr_cases', c);
  }

  static async getCase(id: string): Promise<FDRCase | undefined> {
    return DatabaseService.getById<FDRCase>('fdr_cases', id);
  }

  static async getCasesForUser(userId: string): Promise<FDRCase[]> {
    const all = await DatabaseService.getAll<FDRCase>('fdr_cases');
    return all.filter(c => c.party1Id === userId || c.party2Id === userId);
  }

  static async getPendingScreenings(): Promise<FDRCase[]> {
    const all = await DatabaseService.getAll<FDRCase>('fdr_cases');
    return all.filter(c => c.stage === FDRStage.SCREENING && c.screeningFeePaid && !c.screeningOutcome);
  }

  static async updateOutcome(caseId: string, outcome: ScreeningOutcome): Promise<void> {
    const c = await this.getCase(caseId);
    if (c) {
      c.screeningOutcome = outcome;
      if (outcome === ScreeningOutcome.PROCEED) {
        c.stage = FDRStage.SELECTION;
      } else if (outcome === ScreeningOutcome.NOT_APPROPRIATE) {
        c.stage = FDRStage.CLOSED;
      }
      await DatabaseService.put('fdr_cases', c);
    }
  }

  static async updateSelection(caseId: string, selectionData: FDRCase['selection']): Promise<void> {
    const c = await this.getCase(caseId);
    if (c) {
      c.selection = selectionData;
      if (selectionData.selectedMediatorId) {
        c.stage = FDRStage.PRE_MEDIATION;
      }
      await DatabaseService.put('fdr_cases', c);
    }
  }
}
