import { DatabaseService } from './DatabaseService';
import { IntakeRequest } from '../types';

export class IntakeService {
  /**
   * Create a new intake request (call-back request)
   */
  static async createRequest(
    clientName: string,
    clientEmail: string,
    clientPhone: string,
    description: string
  ): Promise<IntakeRequest> {
    const request: IntakeRequest = {
      id: 'int-' + Math.random().toString(36).substr(2, 9),
      clientName,
      clientEmail,
      clientPhone,
      description,
      status: 'PENDING',
      officerNotes: '',
      createdAt: Date.now(),
      updatedAt: Date.now()
    };
    
    await DatabaseService.put('intakeRequests', request);
    return request;
  }

  /**
   * Get intake request by ID
   */
  static async getById(id: string): Promise<IntakeRequest | undefined> {
    return DatabaseService.getById<IntakeRequest>('intakeRequests', id);
  }

  /**
   * Assign a request to an intake officer
   */
  static async assignToOfficer(requestId: string, officerId: string): Promise<void> {
    const request = await this.getById(requestId);
    if (request) {
      request.assignedOfficerId = officerId;
      request.status = 'IN_PROGRESS';
      request.updatedAt = Date.now();
      await DatabaseService.put('intakeRequests', request);
    }
  }

  /**
   * Add notes to a request
   */
  static async updateNotes(requestId: string, notes: string): Promise<void> {
    const request = await this.getById(requestId);
    if (request) {
      request.officerNotes = notes;
      request.updatedAt = Date.now();
      await DatabaseService.put('intakeRequests', request);
    }
  }

  /**
   * Complete a request
   */
  static async completeRequest(requestId: string, engagementFormId?: string): Promise<void> {
    const request = await this.getById(requestId);
    if (request) {
      request.status = 'COMPLETED';
      request.completedAt = Date.now();
      request.updatedAt = Date.now();
      if (engagementFormId) {
        request.engagementFormId = engagementFormId;
      }
      await DatabaseService.put('intakeRequests', request);
    }
  }

  /**
   * Cancel a request
   */
  static async cancelRequest(requestId: string): Promise<void> {
    const request = await this.getById(requestId);
    if (request) {
      request.status = 'CANCELLED';
      request.updatedAt = Date.now();
      await DatabaseService.put('intakeRequests', request);
    }
  }

  /**
   * Get all pending requests
   */
  static async getPendingRequests(): Promise<IntakeRequest[]> {
    const all = await DatabaseService.getAll<IntakeRequest>('intakeRequests');
    return all.filter(r => r.status === 'PENDING').sort((a, b) => b.createdAt - a.createdAt);
  }

  /**
   * Get requests assigned to an officer
   */
  static async getByOfficer(officerId: string): Promise<IntakeRequest[]> {
    const all = await DatabaseService.getAll<IntakeRequest>('intakeRequests');
    return all.filter(r => r.assignedOfficerId === officerId).sort((a, b) => b.updatedAt - a.updatedAt);
  }

  /**
   * Get all requests
   */
  static async getAllRequests(): Promise<IntakeRequest[]> {
    const all = await DatabaseService.getAll<IntakeRequest>('intakeRequests');
    return all.sort((a, b) => b.createdAt - a.createdAt);
  }
}
