
import { DatabaseService } from './DatabaseService';
import { MediatorProfilePublic, MediatorProfilePrivate, MediatorStatus } from '../types';

export class MediatorService {
  static async getAllMediators(): Promise<MediatorProfilePrivate[]> {
    return DatabaseService.getAll<MediatorProfilePrivate>('mediators');
  }

  static async getPublishedMediators(): Promise<MediatorProfilePublic[]> {
    const all = await this.getAllMediators();
    // Scrub internal fields for public view
    return all
      .filter(m => m.status === MediatorStatus.PUBLISHED)
      .map(m => {
        const { forwardToEmail, ...publicData } = m;
        return publicData;
      });
  }

  static async getMediatorById(id: string, includePrivate: true): Promise<MediatorProfilePrivate | null>;
  static async getMediatorById(id: string, includePrivate?: false): Promise<MediatorProfilePublic | null>;
  static async getMediatorById(id: string, includePrivate = false): Promise<MediatorProfilePublic | MediatorProfilePrivate | null> {
    const med = await DatabaseService.getById<MediatorProfilePrivate>('mediators', id);
    if (!med) return null;
    if (!includePrivate) {
      const { forwardToEmail, ...publicData } = med;
      return publicData;
    }
    return med;
  }

  static async updateProfile(profile: MediatorProfilePrivate): Promise<void> {
    await DatabaseService.put('mediators', profile);
  }

  static async updateStatus(id: string, status: MediatorStatus): Promise<void> {
    const med = await DatabaseService.getById<MediatorProfilePrivate>('mediators', id);
    if (med) {
      med.status = status;
      await DatabaseService.put('mediators', med);
    }
  }

  static async assignAlias(id: string, alias: string): Promise<void> {
    const med = await DatabaseService.getById<MediatorProfilePrivate>('mediators', id);
    if (med) {
      med.publicEmail = alias;
      await DatabaseService.put('mediators', med);
    }
  }
}
