import { DatabaseService } from './DatabaseService';
import { OperatingHours, DaySchedule, DayOfWeek } from '../types';

export class OperatingHoursService {
  /**
   * Create default operating hours for a mediator
   * Default: Monday-Friday 9am-5pm
   */
  static createDefaultSchedule(mediatorId: string): OperatingHours {
    const defaultSchedule: DaySchedule[] = [
      { dayOfWeek: DayOfWeek.MONDAY, isWorking: true, startTime: '09:00', endTime: '17:00' },
      { dayOfWeek: DayOfWeek.TUESDAY, isWorking: true, startTime: '09:00', endTime: '17:00' },
      { dayOfWeek: DayOfWeek.WEDNESDAY, isWorking: true, startTime: '09:00', endTime: '17:00' },
      { dayOfWeek: DayOfWeek.THURSDAY, isWorking: true, startTime: '09:00', endTime: '17:00' },
      { dayOfWeek: DayOfWeek.FRIDAY, isWorking: true, startTime: '09:00', endTime: '17:00' },
      { dayOfWeek: DayOfWeek.SATURDAY, isWorking: false },
      { dayOfWeek: DayOfWeek.SUNDAY, isWorking: false }
    ];

    return {
      id: 'hours-' + mediatorId,
      mediatorId,
      schedule: defaultSchedule,
      timezone: 'Australia/Sydney',
      createdAt: Date.now(),
      updatedAt: Date.now()
    };
  }

  /**
   * Get or create operating hours for a mediator
   */
  static async getOperatingHours(mediatorId: string): Promise<OperatingHours> {
    const existing = await DatabaseService.getById<OperatingHours>('operatingHours', 'hours-' + mediatorId);
    
    if (existing) {
      return existing;
    }

    // Create default schedule
    const defaultSchedule = this.createDefaultSchedule(mediatorId);
    await DatabaseService.put('operatingHours', defaultSchedule);
    return defaultSchedule;
  }

  /**
   * Update operating hours
   */
  static async updateOperatingHours(
    mediatorId: string,
    schedule: DaySchedule[],
    timezone?: string
  ): Promise<void> {
    const hours = await this.getOperatingHours(mediatorId);
    
    hours.schedule = schedule;
    if (timezone) hours.timezone = timezone;
    hours.updatedAt = Date.now();
    
    await DatabaseService.put('operatingHours', hours);
  }

  /**
   * Update a specific day's schedule
   */
  static async updateDay(
    mediatorId: string,
    dayOfWeek: DayOfWeek,
    isWorking: boolean,
    startTime?: string,
    endTime?: string
  ): Promise<void> {
    const hours = await this.getOperatingHours(mediatorId);
    
    const daySchedule = hours.schedule.find(d => d.dayOfWeek === dayOfWeek);
    if (daySchedule) {
      daySchedule.isWorking = isWorking;
      daySchedule.startTime = startTime;
      daySchedule.endTime = endTime;
    }
    
    hours.updatedAt = Date.now();
    await DatabaseService.put('operatingHours', hours);
  }

  /**
   * Check if mediator is working on a specific day
   */
  static async isWorkingDay(mediatorId: string, date: Date): Promise<boolean> {
    const hours = await this.getOperatingHours(mediatorId);
    const dayOfWeek = date.getDay(); // 0=Sunday, 1=Monday, etc.
    
    // Convert JS day (0=Sun) to our enum (0=Mon)
    const adjustedDay = dayOfWeek === 0 ? 6 : dayOfWeek - 1;
    
    const daySchedule = hours.schedule.find(d => d.dayOfWeek === adjustedDay);
    return daySchedule?.isWorking ?? false;
  }

  /**
   * Check if a specific time falls within operating hours
   */
  static async isWithinOperatingHours(
    mediatorId: string,
    date: Date,
    time: string // "14:30" format
  ): Promise<boolean> {
    // First check if working day
    const isWorking = await this.isWorkingDay(mediatorId, date);
    if (!isWorking) return false;

    const hours = await this.getOperatingHours(mediatorId);
    const dayOfWeek = date.getDay() === 0 ? 6 : date.getDay() - 1;
    
    const daySchedule = hours.schedule.find(d => d.dayOfWeek === dayOfWeek);
    if (!daySchedule?.startTime || !daySchedule?.endTime) return false;

    // Compare times (simple string comparison works for HH:MM format)
    return time >= daySchedule.startTime && time <= daySchedule.endTime;
  }

  /**
   * Get formatted schedule for display
   */
  static getDayName(dayOfWeek: DayOfWeek): string {
    const names = ['Monday', 'Tuesday', 'Wednesday', 'Thursday', 'Friday', 'Saturday', 'Sunday'];
    return names[dayOfWeek];
  }

  /**
   * Format time for display (24h to 12h)
   */
  static formatTime(time: string): string {
    const [hours, minutes] = time.split(':');
    const hour = parseInt(hours);
    const ampm = hour >= 12 ? 'PM' : 'AM';
    const displayHour = hour === 0 ? 12 : hour > 12 ? hour - 12 : hour;
    return `${displayHour}:${minutes} ${ampm}`;
  }

  /**
   * Get summary of working days
   */
  static getWorkingSummary(hours: OperatingHours): string {
    const workingDays = hours.schedule.filter(d => d.isWorking);
    
    if (workingDays.length === 0) return 'Not working';
    if (workingDays.length === 7) return 'Every day';
    
    // Get day names
    const dayNames = workingDays.map(d => this.getDayName(d.dayOfWeek));
    
    // Check for consecutive weekdays
    const isWeekdays = workingDays.every(d => d.dayOfWeek <= DayOfWeek.FRIDAY);
    if (isWeekdays && workingDays.length === 5) {
      return 'Monday - Friday';
    }
    
    // Check for weekend
    const isWeekend = workingDays.every(d => d.dayOfWeek >= DayOfWeek.SATURDAY);
    if (isWeekend && workingDays.length === 2) {
      return 'Saturday - Sunday';
    }
    
    // List days
    if (dayNames.length <= 3) {
      return dayNames.join(', ');
    }
    
    return `${dayNames.length} days per week`;
  }

  /**
   * Copy schedule to all weekdays
   */
  static async copyToWeekdays(
    mediatorId: string,
    startTime: string,
    endTime: string
  ): Promise<void> {
    const hours = await this.getOperatingHours(mediatorId);
    
    // Update Mon-Fri
    for (let day = DayOfWeek.MONDAY; day <= DayOfWeek.FRIDAY; day++) {
      const daySchedule = hours.schedule.find(d => d.dayOfWeek === day);
      if (daySchedule) {
        daySchedule.isWorking = true;
        daySchedule.startTime = startTime;
        daySchedule.endTime = endTime;
      }
    }
    
    hours.updatedAt = Date.now();
    await DatabaseService.put('operatingHours', hours);
  }

  /**
   * Copy schedule to all days
   */
  static async copyToAllDays(
    mediatorId: string,
    startTime: string,
    endTime: string
  ): Promise<void> {
    const hours = await this.getOperatingHours(mediatorId);
    
    hours.schedule.forEach(day => {
      day.isWorking = true;
      day.startTime = startTime;
      day.endTime = endTime;
    });
    
    hours.updatedAt = Date.now();
    await DatabaseService.put('operatingHours', hours);
  }
}
