
import { DatabaseService } from './DatabaseService';
import { 
  UserRole, 
  MediatorStatus, 
  HourlyRateBasis, 
  User, 
  MediationCategory, 
  MediatorProfilePrivate 
} from '../types';

export const SEEDED_CATEGORIES: MediationCategory[] = [
  { id: 'cat-family', name: 'Family Law & Parenting' },
  { id: 'cat-workplace', name: 'Workplace & Employment' },
  { id: 'cat-commercial', name: 'Commercial & Business' },
  { id: 'cat-estate', name: 'Wills & Estates' },
  { id: 'cat-property', name: 'Property & Real Estate' },
  { id: 'cat-neighborhood', name: 'Neighborhood & Community' },
  { id: 'cat-construction', name: 'Construction & Building' },
  { id: 'cat-elder', name: 'Elder Mediation' },
  { id: 'cat-medical', name: 'Medical & Health' },
  { id: 'cat-franchise', name: 'Franchise Disputes' }
];

const DEMO_USERS: User[] = [
  { id: 'u-admin', email: 'admin@medi8.com.au', role: UserRole.ADMIN, displayName: 'System Admin' },
  { id: 'u-mediator1', email: 'mediator1@medi8r.com.au', role: UserRole.MEDIATOR, displayName: 'Demo Mediator', profileId: 'med-demo' },
  { id: 'u-m1', email: 'jane@medi8r.com.au', role: UserRole.MEDIATOR, displayName: 'Jane Smith', profileId: 'm-1' },
  { id: 'u-m2', email: 'robert@medi8r.com.au', role: UserRole.MEDIATOR, displayName: 'Robert Chen', profileId: 'm-2' },
  { id: 'u-m3', email: 'alice@medi8r.com.au', role: UserRole.MEDIATOR, displayName: 'Alice Waters', profileId: 'm-3' },
  { id: 'u-m4', email: 'david@medi8r.com.au', role: UserRole.MEDIATOR, displayName: 'David Miller', profileId: 'm-4' },
  { id: 'u-m5', email: 'sarah@medi8r.com.au', role: UserRole.MEDIATOR, displayName: 'Sarah Johnson', profileId: 'm-5' },
  { id: 'u-m6', email: 'michael@medi8r.com.au', role: UserRole.MEDIATOR, displayName: 'Michael Brown', profileId: 'm-6' },
  { id: 'u-m7', email: 'elena@medi8r.com.au', role: UserRole.MEDIATOR, displayName: 'Elena Rodriguez', profileId: 'm-7' },
  { id: 'u-m8', email: 'kevin@medi8r.com.au', role: UserRole.MEDIATOR, displayName: 'Kevin Lee', profileId: 'm-8' },
  { id: 'u-public', email: 'public@demo.com', role: UserRole.PUBLIC, displayName: 'John Public' }
];

// Add 20 more users for the extra mediators
for (let i = 9; i <= 28; i++) {
  DEMO_USERS.push({
    id: `u-m${i}`,
    email: `mediator${i}@medi8r.com.au`,
    role: UserRole.MEDIATOR,
    displayName: `Mediator ${i}`,
    profileId: `m-${i}`
  });
}

const DEMO_MEDIATORS: MediatorProfilePrivate[] = [
  {
    id: 'med-demo',
    userId: 'u-mediator1',
    displayName: 'Demo Mediator',
    photoUrl: 'https://picsum.photos/seed/demo/200/200',
    headline: 'Test Mediator for Platform Development',
    bio: 'Demo mediator profile for testing all platform features including pricing, availability, and promotional offers.',
    categories: ['cat-family', 'cat-commercial'],
    locations: ['Sydney', 'Online'],
    onlineAvailable: true,
    publicEmail: 'demo@medi8r.com.au',
    forwardToEmail: 'mediator1@medi8r.com.au',
    status: MediatorStatus.PUBLISHED,
    pricing: {
      preMediationFeeCents: 35000,
      hourlyRateCents: 45000,
      hourlyRateBasis: HourlyRateBasis.TOTAL,
      travelEnabled: false,
      includedKm: 0,
      perKmCents: 0,
      accommodationEnabled: false,
      hasPromotionalOffer: false,
      promotionalPreMediationFeeCents: 0,
      promotionalOfferDescription: ''
    },
    qualifications: ['NMAS Accredited', 'LLM (Dispute Resolution)'],
    experience: [{ title: 'Demo Mediator', period: '2020-Present', description: 'Testing platform features' }],
    optInFixedFee: false,
    activePackages: []
  },
  {
    id: 'm-1',
    userId: 'u-m1',
    displayName: 'Jane Smith',
    photoUrl: 'https://picsum.photos/seed/jane/200/200',
    headline: 'Specialist in Family Law and Parenting Disputes',
    bio: 'Jane has over 15 years experience as a solicitor and mediator helping families transition through difficult separations with dignity.',
    categories: ['cat-family', 'cat-property', 'cat-estate'],
    locations: ['Sydney', 'Online'],
    onlineAvailable: true,
    publicEmail: 'jane.smith@medi8r.com.au',
    forwardToEmail: 'jane.mediator@gmail.com',
    status: MediatorStatus.PUBLISHED,
    pricing: {
      preMediationFeeCents: 45000,
      hourlyRateCents: 25000,
      hourlyRateBasis: HourlyRateBasis.PER_PARTY,
      travelEnabled: false,
      includedKm: 0,
      perKmCents: 0,
      accommodationEnabled: false
    },
    qualifications: ['LLM (DR)', 'NMAS Accredited'],
    experience: [{ title: 'Principal Mediator', period: '2015-Present', description: 'Private practice' }],
    // Fix: Missing properties from type 'MediatorProfilePrivate'
    optInFixedFee: true,
    activePackages: ['ff-parenting-express']
  },
  {
    id: 'm-2',
    userId: 'u-m2',
    displayName: 'Robert Chen',
    photoUrl: 'https://picsum.photos/seed/rob/200/200',
    headline: 'Commercial Mediator for Complex Business Conflict',
    bio: 'Robert focuses on high-stakes resolution of commercial disputes to minimize business disruption and legal costs.',
    categories: ['cat-commercial', 'cat-franchise', 'cat-construction'],
    locations: ['Melbourne', 'National'],
    onlineAvailable: true,
    publicEmail: 'robert.chen@medi8r.com.au',
    forwardToEmail: 'rob.chen.business@gmail.com',
    status: MediatorStatus.PUBLISHED,
    pricing: {
      preMediationFeeCents: 120000,
      hourlyRateCents: 50000,
      hourlyRateBasis: HourlyRateBasis.PER_PARTY,
      travelEnabled: true,
      includedKm: 100,
      perKmCents: 95,
      accommodationEnabled: true,
      estimatedPerNightCents: 25000
    },
    qualifications: ['MBA', 'JD', 'International Arbitrator'],
    experience: [{ title: 'Commercial Lead', period: '2010-2024', description: 'Big Four consulting dispute lead' }],
    // Fix: Missing properties from type 'MediatorProfilePrivate'
    optInFixedFee: true,
    activePackages: ['ff-business-starter']
  },
  {
    id: 'm-3',
    userId: 'u-m3',
    displayName: 'Alice Waters',
    photoUrl: 'https://picsum.photos/seed/alice/200/200',
    headline: 'Community and Neighborhood Resolution Specialist',
    bio: 'Providing affordable, accessible mediation services for local disputes and community disagreements.',
    categories: ['cat-neighborhood', 'cat-property'],
    locations: ['Brisbane', 'Online'],
    onlineAvailable: true,
    publicEmail: 'alice.waters@medi8r.com.au',
    forwardToEmail: 'alice.w@outlook.com',
    status: MediatorStatus.PUBLISHED,
    pricing: {
      preMediationFeeCents: 15000,
      hourlyRateCents: 15000,
      hourlyRateBasis: HourlyRateBasis.TOTAL,
      travelEnabled: false,
      includedKm: 0,
      perKmCents: 0,
      accommodationEnabled: false
    },
    qualifications: ['Bachelor of Social Work', 'NMAS Accredited'],
    experience: [],
    // Fix: Missing properties from type 'MediatorProfilePrivate'
    optInFixedFee: false,
    activePackages: []
  },
  {
    id: 'm-4',
    userId: 'u-m4',
    displayName: 'David Miller',
    photoUrl: 'https://picsum.photos/seed/david/200/200',
    headline: 'Workplace Harmony and Industrial Relations',
    bio: 'Dedicated to resolving workplace conflict and improving organisational culture through proactive mediation.',
    categories: ['cat-workplace', 'cat-commercial'],
    locations: ['Perth', 'Adelaide'],
    onlineAvailable: false,
    publicEmail: 'david.miller@medi8r.com.au',
    forwardToEmail: 'd.miller.hr@gmail.com',
    status: MediatorStatus.PUBLISHED,
    pricing: {
      preMediationFeeCents: 60000,
      hourlyRateCents: 50000,
      hourlyRateBasis: HourlyRateBasis.TOTAL,
      travelEnabled: true,
      includedKm: 30,
      perKmCents: 80,
      accommodationEnabled: false
    },
    qualifications: ['HR Management Diploma', 'Conflict Coach'],
    experience: [{ title: 'HR Director', period: '2005-2020', description: 'Global Logistics' }],
    // Fix: Missing properties from type 'MediatorProfilePrivate'
    optInFixedFee: true,
    activePackages: ['ff-workplace-checkin']
  },
  {
    id: 'm-5',
    userId: 'u-m5',
    displayName: 'Sarah Johnson',
    photoUrl: 'https://picsum.photos/seed/sarah/200/200',
    headline: 'Empathetic Resolution for Estates and Families',
    bio: 'Sarah bridges the gap in emotional disputes involving inheritance and family transition.',
    categories: ['cat-estate', 'cat-family', 'cat-elder'],
    locations: ['Canberra', 'Online'],
    onlineAvailable: true,
    publicEmail: 's.johnson@medi8r.com.au',
    forwardToEmail: 'sarah.johnson.law@gmail.com',
    status: MediatorStatus.PUBLISHED,
    pricing: {
      preMediationFeeCents: 30000,
      hourlyRateCents: 15000,
      hourlyRateBasis: HourlyRateBasis.PER_PARTY,
      travelEnabled: false,
      includedKm: 0,
      perKmCents: 0,
      accommodationEnabled: false
    },
    qualifications: ['Bachelor of Laws', 'Postgrad Dispute Resolution'],
    experience: [],
    // Fix: Missing properties from type 'MediatorProfilePrivate'
    optInFixedFee: false,
    activePackages: []
  },
  {
    id: 'm-6',
    userId: 'u-m6',
    displayName: 'Michael Brown',
    photoUrl: 'https://picsum.photos/seed/mike/200/200',
    headline: 'Construction and Infrastructure Dispute Expert',
    bio: 'Expert technical mediation for building, construction, and major infrastructure project delays.',
    categories: ['cat-construction', 'cat-commercial', 'cat-property'],
    locations: ['Sydney', 'National'],
    onlineAvailable: true,
    publicEmail: 'm.brown@medi8r.com.au',
    forwardToEmail: 'michael.brown.const@outlook.com',
    status: MediatorStatus.PUBLISHED,
    pricing: {
      preMediationFeeCents: 180000,
      hourlyRateCents: 80000,
      hourlyRateBasis: HourlyRateBasis.TOTAL,
      travelEnabled: true,
      includedKm: 50,
      perKmCents: 120,
      accommodationEnabled: true,
      estimatedPerNightCents: 30000
    },
    qualifications: ['Civil Engineer', 'Adjudicator'],
    experience: [{ title: 'Chief Engineer', period: '1998-2018', description: 'State Transport' }],
    // Fix: Missing properties from type 'MediatorProfilePrivate'
    optInFixedFee: false,
    activePackages: []
  },
  {
    id: 'm-7',
    userId: 'u-m7',
    displayName: 'Elena Rodriguez',
    photoUrl: 'https://picsum.photos/seed/elena/200/200',
    headline: 'Healthcare and Elder Care Neutral',
    bio: 'Navigating sensitive decisions in medical practice and elder care environments with neutrality and care.',
    categories: ['cat-medical', 'cat-elder', 'cat-estate'],
    locations: ['Hobart', 'Online'],
    onlineAvailable: true,
    publicEmail: 'elena.r@medi8r.com.au',
    forwardToEmail: 'elena.rodriguez@gmail.com',
    status: MediatorStatus.PUBLISHED,
    pricing: {
      preMediationFeeCents: 50000,
      hourlyRateCents: 30000,
      hourlyRateBasis: HourlyRateBasis.PER_PARTY,
      travelEnabled: false,
      includedKm: 0,
      perKmCents: 0,
      accommodationEnabled: false
    },
    qualifications: ['Masters in Health Administration', 'NMAS Accredited'],
    experience: [],
    // Fix: Missing properties from type 'MediatorProfilePrivate'
    optInFixedFee: false,
    activePackages: []
  },
  {
    id: 'm-8',
    userId: 'u-m8',
    displayName: 'Kevin Lee',
    photoUrl: 'https://picsum.photos/seed/kevin/200/200',
    headline: 'Practical Small Business and Workplace Solutions',
    bio: 'Pragmatic, no-nonsense mediation for small business and workplace conflicts.',
    categories: ['cat-neighborhood', 'cat-workplace', 'cat-franchise'],
    locations: ['Darwin', 'Online'],
    onlineAvailable: true,
    publicEmail: 'kevin.lee@medi8r.com.au',
    forwardToEmail: 'kevin.lee.resolutions@gmail.com',
    status: MediatorStatus.PUBLISHED,
    pricing: {
      preMediationFeeCents: 20000,
      hourlyRateCents: 20000,
      hourlyRateBasis: HourlyRateBasis.TOTAL,
      travelEnabled: false,
      includedKm: 0,
      perKmCents: 0,
      accommodationEnabled: false
    },
    qualifications: ['B.A. Psychology', 'Cert IV Mediation'],
    experience: [],
    // Fix: Missing properties from type 'MediatorProfilePrivate'
    optInFixedFee: false,
    activePackages: []
  }
];

// Generation of 20 low-cost mediators across Australia
const extraLocations = ['Gold Coast', 'Newcastle', 'Wollongong', 'Geelong', 'Townsville', 'Cairns', 'Toowoomba', 'Ballarat', 'Bendigo', 'Launceston', 'Mackay', 'Rockhampton', 'Bunbury', 'Coffs Harbour', 'Bundaberg', 'Wagga Wagga', 'Hervey Bay', 'Shepparton', 'Albury', 'Port Macquarie'];
const firstNames = ['James', 'Mary', 'John', 'Patricia', 'Robert', 'Jennifer', 'Michael', 'Linda', 'William', 'Elizabeth', 'David', 'Barbara', 'Richard', 'Susan', 'Joseph', 'Jessica', 'Thomas', 'Sarah', 'Charles', 'Karen'];
const lastNames = ['Smith', 'Johnson', 'Williams', 'Brown', 'Jones', 'Garcia', 'Miller', 'Davis', 'Rodriguez', 'Martinez', 'Hernandez', 'Lopez', 'Gonzalez', 'Wilson', 'Anderson', 'Thomas', 'Taylor', 'Moore', 'Jackson', 'Martin'];

for (let i = 0; i < 20; i++) {
  const mediatorId = `m-${i + 9}`;
  const userId = `u-m${i + 9}`;
  const firstName = firstNames[i];
  const lastName = lastNames[i];
  const location = extraLocations[i];
  const cats = SEEDED_CATEGORIES.sort(() => 0.5 - Math.random()).slice(0, Math.floor(Math.random() * 3) + 2).map(c => c.id);
  
  // Budget pricing focused
  const isBudget = Math.random() > 0.4;
  const preFee = isBudget ? Math.floor(Math.random() * 200) + 100 : Math.floor(Math.random() * 400) + 200;
  const hourly = isBudget ? Math.floor(Math.random() * 100) + 100 : Math.floor(Math.random() * 150) + 150;

  DEMO_MEDIATORS.push({
    id: mediatorId,
    userId: userId,
    displayName: `${firstName} ${lastName}`,
    photoUrl: `https://picsum.photos/seed/med${i+9}/200/200`,
    headline: `Professional Mediation in ${location}`,
    bio: `${firstName} provides expert conflict resolution services for ${cats.length} different sectors, ensuring a calm and efficient process for all parties in ${location}.`,
    categories: cats,
    locations: [location, 'Online'],
    onlineAvailable: Math.random() > 0.1,
    publicEmail: `${firstName.toLowerCase()}.${lastName.toLowerCase()}@medi8r.com.au`,
    forwardToEmail: `${firstName.toLowerCase()}${i}@gmail.com`,
    status: MediatorStatus.PUBLISHED,
    pricing: {
      preMediationFeeCents: preFee * 100,
      hourlyRateCents: hourly * 100,
      hourlyRateBasis: Math.random() > 0.5 ? HourlyRateBasis.TOTAL : HourlyRateBasis.PER_PARTY,
      travelEnabled: Math.random() > 0.7,
      includedKm: 20,
      perKmCents: 80,
      accommodationEnabled: false
    },
    qualifications: ['NMAS Accredited', 'Conflict Resolution Specialist'],
    experience: [],
    // Fix: Missing properties from type 'MediatorProfilePrivate'
    optInFixedFee: false,
    activePackages: []
  });
}

export class SeedService {
  static async seedDatabase() {
    const categories = await DatabaseService.getAll<MediationCategory>('categories');
    if (categories.length === 0) {
      for (const cat of SEEDED_CATEGORIES) await DatabaseService.put('categories', cat);
      for (const user of DEMO_USERS) await DatabaseService.put('users', user);
      for (const med of DEMO_MEDIATORS) await DatabaseService.put('mediators', med);
      console.log('Database seeded successfully');
    }
  }

  static async resetDatabase() {
    await DatabaseService.clearAll();
    await this.seedDatabase();
    window.location.reload();
  }
}
