/**
 * StripeService - Payment processing via Stripe Connect
 * Handles checkout sessions and payment intents
 */

import { loadStripe } from '@stripe/stripe-js';

const stripePromise = loadStripe(import.meta.env.VITE_STRIPE_PUBLISHABLE_KEY);

export interface CheckoutSessionData {
  mediatorId: string;
  mediatorName: string;
  clientName: string;
  clientEmail: string;
  amount: number; // in cents
  description: string;
  metadata: Record<string, string>;
}

export class StripeService {
  /**
   * Create a Stripe Checkout session for pre-mediation fee
   */
  static async createCheckoutSession(data: CheckoutSessionData): Promise<string> {
    try {
      const apiUrl = import.meta.env.VITE_PAYMENT_API_URL || 'http://localhost:3001';
      
      const response = await fetch(`${apiUrl}/api/create-checkout-session`, {
        method: 'POST',
        headers: {
          'Content-Type': 'application/json',
        },
        body: JSON.stringify(data)
      });

      if (!response.ok) {
        const error = await response.json();
        throw new Error(error.message || 'Failed to create checkout session');
      }

      const { sessionId } = await response.json();
      
      // Redirect to Stripe Checkout
      const stripe = await stripePromise;
      if (!stripe) {
        throw new Error('Stripe failed to load');
      }

      const { error } = await stripe.redirectToCheckout({ sessionId });
      
      if (error) {
        throw error;
      }

      return sessionId;
      
    } catch (error) {
      console.error('Stripe checkout error:', error);
      throw error;
    }
  }

  /**
   * Format amount in AUD for display
   */
  static formatAud(cents: number): string {
    return new Intl.NumberFormat('en-AU', {
      style: 'currency',
      currency: 'AUD'
    }).format(cents / 100);
  }

  /**
   * Calculate platform fee (10%)
   */
  static calculatePlatformFee(totalCents: number): number {
    const feePercentage = parseInt(import.meta.env.VITE_PLATFORM_FEE_PERCENTAGE || '10');
    return Math.round(totalCents * (feePercentage / 100));
  }
}
