
export enum UserRole {
  ADMIN = 'ADMIN',
  MEDIATOR = 'MEDIATOR',
  STAFF = 'STAFF', // Intake officers
  PUBLIC = 'PUBLIC'
}

export enum MediatorStatus {
  DRAFT = 'DRAFT',
  PENDING_APPROVAL = 'PENDING_APPROVAL',
  PUBLISHED = 'PUBLISHED',
  SUSPENDED = 'SUSPENDED'
}

export enum BookingStatus {
  PENDING = 'PENDING',
  DEPOSIT_PAID = 'DEPOSIT_PAID',
  PROCEED_APPROVED = 'PROCEED_APPROVED',
  BALANCE_PAID = 'BALANCE_PAID',
  DECLINED = 'DECLINED',
  CANCELLED = 'CANCELLED'
}

export enum FDRStage {
  SCREENING = 'SCREENING',
  SELECTION = 'SELECTION',
  PRE_MEDIATION = 'PRE_MEDIATION',
  MEDIATION = 'MEDIATION',
  CLOSED = 'CLOSED'
}

export enum ScreeningOutcome {
  PROCEED = 'PROCEED',
  ESCALATE = 'ESCALATE',
  NOT_APPROPRIATE = 'NOT_APPROPRIATE'
}

export enum ESignStatus {
  NOT_SENT = 'NOT_SENT',
  SENT = 'SENT',
  SIGNED = 'SIGNED',
  EXPIRED = 'EXPIRED'
}

export enum PaymentKind {
  DEPOSIT = 'DEPOSIT',
  BALANCE = 'BALANCE',
  SCREENING_FEE = 'SCREENING_FEE'
}

export enum HourlyRateBasis {
  PER_PARTY = 'PER_PARTY',
  TOTAL = 'TOTAL'
}

// Added PriceBand enum to resolve import errors in lib/pricing.ts, lib/suggest.ts, and pages/public/MediatorDirectory.tsx
export enum PriceBand {
  BUDGET = 'BUDGET',
  STANDARD = 'STANDARD',
  PREMIUM = 'PREMIUM'
}

export interface FixedFeePackage {
  id: string;
  name: string;
  description: string;
  durationHours: number;
  totalFeeCents: number;
  category: string;
}

export interface User {
  id: string;
  email: string;
  role: UserRole;
  displayName: string;
  profileId?: string;
}

export interface FDRCase {
  id: string;
  party1Id: string; // User ID
  party1Email: string;
  party2Email?: string;
  party2Id?: string; // Set once invited/joined
  disputeType: 'PARENTING' | 'PROPERTY' | 'BOTH';
  stage: FDRStage;
  timestamp: number;
  
  // Stage 1: Screening
  screeningOutcome?: ScreeningOutcome;
  screeningFeePaid: boolean;
  declarations: {
    familyViolence: boolean;
    interventionOrders: boolean;
    pendingProceedings: boolean;
  };
  documents: { type: string; url: string; name: string }[];
  
  // Stage 2: Selection
  selection: {
    round: 1 | 2;
    p1ProposedIds: string[];
    p2ProposedIds: string[];
    selectedMediatorId?: string;
    selectionFailed: boolean;
  };

  // Stage 3 & 4
  preMediationCompleted: boolean;
  bookingId?: string; // Links to standard BookingRequest once finalized
}

export interface MediatorProfilePublic {
  id: string;
  userId: string;
  displayName: string;
  photoUrl: string;
  headline: string;
  bio: string;
  categories: string[];
  locations: string[];
  onlineAvailable: boolean;
  publicEmail?: string; 
  status: MediatorStatus;
  pricing: PricingModel;
  qualifications: string[];
  experience: { title: string; period: string; description: string }[];
  optInFixedFee: boolean;
  activePackages: string[];
}

export interface MediatorProfilePrivate extends MediatorProfilePublic {
  forwardToEmail: string;
}

export interface BookingRequest {
  id: string;
  mediatorId: string;
  clientName: string;
  clientEmail: string;
  notes: string;
  status: BookingStatus;
  feeTotalCents?: number;
  depositAmountCents: number;
  timestamp: number;
  isFixedFee: boolean;
  packageId?: string;
  proposedDateTime?: number;
  esignStatus: ESignStatus;
  esignEnvelopeId?: string;
}

export interface PaymentEvent {
  id: string;
  bookingId: string; // or fdrCaseId
  kind: PaymentKind;
  amountCents: number;
  timestamp: number;
}

export interface MediationCategory {
  id: string;
  name: string;
}

export interface PricingModel {
  preMediationFeeCents: number;
  hourlyRateCents: number;
  hourlyRateBasis: HourlyRateBasis;
  travelEnabled: boolean;
  includedKm: number;
  perKmCents: number;
  accommodationEnabled: boolean;
  estimatedPerNightCents?: number;
  // Added travelTimeHourlyRateCents to resolve error in lib/pricing.ts (line 24)
  travelTimeHourlyRateCents?: number;
  
  // Promotional Offers
  hasPromotionalOffer?: boolean;
  promotionalPreMediationFeeCents?: number; // Discounted rate
  promotionalOfferDescription?: string; // e.g., "First consultation discounted"
}

// Added AuditLogEntry to resolve import errors in services/AuditService.ts and pages/admin/AuditLogPage.tsx
export interface AuditLogEntry {
  id: string;
  timestamp: number;
  actorId: string;
  actorEmail: string;
  action: string;
  targetId: string;
  details: string;
}

// Added Enquiry to resolve import error in services/EnquiryService.ts
export interface Enquiry {
  id: string;
  mediatorId: string;
  clientName: string;
  clientEmail: string;
  subject: string;
  message: string;
  timestamp: number;
}

// Phase 1: New Engagement Workflow Types

export enum EngagementStage {
  CONTACT_INFO = 'CONTACT_INFO',
  DISPUTE_DETAILS = 'DISPUTE_DETAILS',
  LEGAL_CONTEXT = 'LEGAL_CONTEXT',
  REVIEW = 'REVIEW',
  PAYMENT = 'PAYMENT',
  COMPLETED = 'COMPLETED',
  ABANDONED = 'ABANDONED'
}

export enum DisputeType {
  FAMILY_PARENTING = 'FAMILY_PARENTING',
  FAMILY_PROPERTY = 'FAMILY_PROPERTY',
  COMMERCIAL = 'COMMERCIAL',
  WORKPLACE = 'WORKPLACE',
  COMMUNITY = 'COMMUNITY',
  OTHER = 'OTHER'
}

export interface EngagementForm {
  id: string;
  mediatorId: string;
  stage: EngagementStage;
  
  // Contact Info
  clientName: string;
  clientEmail: string;
  clientPhone: string;
  preferredLocation: string; // or "Online"
  preferInPerson: boolean;
  
  // Dispute Details
  disputeType: DisputeType;
  disputeDuration: string; // e.g., "6 months", "2 years"
  otherPartyName: string;
  disputeDescription: string;
  
  // Legal Context
  hasCourtOrders: boolean;
  hasSafetyOrders: boolean;
  hasContracts: boolean;
  contextDetails: string;
  
  // Special Considerations
  needsDisabilityAccommodation: boolean;
  culturalOrLanguageNeeds: string;
  safetyConcerns: string;
  
  // Metadata
  createdAt: number;
  updatedAt: number;
  lastActivityAt: number;
  completedAt?: number;
  
  // Tracking
  estimatedFeeCents: number;
}

export interface IntakeRequest {
  id: string;
  clientName: string;
  clientEmail: string;
  clientPhone: string;
  description: string;
  status: 'PENDING' | 'IN_PROGRESS' | 'COMPLETED' | 'CANCELLED';
  assignedOfficerId?: string;
  officerNotes: string;
  createdAt: number;
  updatedAt: number;
  completedAt?: number;
  engagementFormId?: string; // If converted to engagement
}

export interface AbandonedEngagement {
  id: string;
  formId: string;
  clientEmail: string;
  clientName: string;
  mediatorId: string;
  lastStage: EngagementStage;
  lastActivityAt: number;
  followUpsSent: number;
  lastFollowUpAt?: number;
  recovered: boolean;
  recoveredAt?: number;
}

// Availability Management
export enum AvailabilityStatus {
  AVAILABLE = 'AVAILABLE',
  UNAVAILABLE = 'UNAVAILABLE'
}

export interface AvailabilityPeriod {
  id: string;
  mediatorId: string;
  startDate: number; // timestamp
  endDate: number; // timestamp
  reason?: string; // "Holiday", "Illness", "Conference", etc.
  notes?: string; // Optional internal notes
  createdAt: number;
  updatedAt: number;
}

export interface MediatorAvailability {
  mediatorId: string;
  status: AvailabilityStatus;
  currentPeriod?: AvailabilityPeriod; // If unavailable now
  upcomingPeriods: AvailabilityPeriod[]; // Future unavailability
  lastUpdated: number;
}

// Operating Hours Management
export enum DayOfWeek {
  MONDAY = 0,
  TUESDAY = 1,
  WEDNESDAY = 2,
  THURSDAY = 3,
  FRIDAY = 4,
  SATURDAY = 5,
  SUNDAY = 6
}

export interface DaySchedule {
  dayOfWeek: DayOfWeek;
  isWorking: boolean;
  startTime?: string; // "09:00"
  endTime?: string;   // "17:00"
}

export interface OperatingHours {
  id: string;
  mediatorId: string;
  schedule: DaySchedule[]; // 7 days
  timezone: string; // e.g., "Australia/Sydney"
  createdAt: number;
  updatedAt: number;
}

